package nexcore.sprout.spring.boot.autoconfigure.resourcebundle;

import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import nexcore.sprout.foundry.resourcebundle.code.DBReloadableResourceBundleCodeSource;
import nexcore.sprout.foundry.resourcebundle.code.ReloadableResourceBundleCodeSource;
import nexcore.sprout.spring.boot.autoconfigure.SproutAutoConfigConst;
import nexcore.sprout.spring.boot.autoconfigure.jdbc.SproutDataSourceAutoConfiguration;
import nexcore.sprout.spring.boot.autoconfigure.resourcebundle.CodeSourceProperties.CodeDataSourceProperties;

@Configuration
@EnableConfigurationProperties(CodeSourceProperties.class)
@AutoConfigureBefore(CodeSourceAccessorAutoConfiguration.class)
@AutoConfigureAfter(SproutDataSourceAutoConfiguration.class)
@ConditionalOnProperty(prefix = SproutAutoConfigConst.PREFIX_SPROUT_CODE_RESOURCE, name = "basename")
public class CodeSourceAutoConfiguration implements ApplicationContextAware {
	
	private static final Log logger = LogFactory.getLog(CodeSourceAutoConfiguration.class);
	
	@Autowired
	private ApplicationContext applicationContext;
	
	@Autowired
	private CodeSourceProperties codeSourceProperties;

	@Bean
	public ReloadableResourceBundleCodeSource codeSource() {
		ReloadableResourceBundleCodeSource codeSource = new ReloadableResourceBundleCodeSource();
		codeSource.setBasenames(StringUtils.commaDelimitedListToStringArray(StringUtils.trimAllWhitespace(codeSourceProperties.getBasename())));
		
		if(codeSourceProperties.getDefaultEncoding() != null)
			codeSource.setDefaultEncoding(codeSourceProperties.getDefaultEncoding());
		if(codeSourceProperties.isFallbackToSystemLocale())
			codeSource.setFallbackToSystemLocale(codeSourceProperties.isFallbackToSystemLocale());
		if(codeSourceProperties.getDefaultFilePostFix() != null)
			codeSource.setDefaultFilePostFix(codeSourceProperties.getDefaultFilePostFix());
		if(codeSourceProperties.getName() != null)
			codeSource.setBeanName(codeSourceProperties.getName());
		if(codeSourceProperties.getCacheSeconds() != 0)			
			codeSource.setCacheSeconds(codeSourceProperties.getCacheSeconds());
		if(codeSourceProperties.getOrder() != Integer.MAX_VALUE)			
			codeSource.setOrder(codeSourceProperties.getOrder());
		
		try {
			DBReloadableResourceBundleCodeSource codeSourceForDb = this.applicationContext.getBean(DBReloadableResourceBundleCodeSource.class);
			codeSource.setParentCodeSource(codeSourceForDb);
			logger.info("====== CodeSource For ParentMessageSource is created.");
		} catch (NoSuchBeanDefinitionException ex) {
			logger.info("====== CodeSource For ParentMessageSource is not created.");
		}
		
		logger.info("====== Creating ReloadableResourceBundleCodeSource Bean: " + codeSource.getClass().toString());
		return codeSource;
	}
	
	@Configuration
	@EnableConfigurationProperties(CodeDataSourceProperties.class)
	@AutoConfigureBefore(CodeSourceAutoConfiguration.class)
	@AutoConfigureAfter(SproutDataSourceAutoConfiguration.class)
	@ConditionalOnProperty(prefix = SproutAutoConfigConst.PREFIX_SPROUT_CODE_DATASOURCE, name = "enabled", havingValue = "true")
	protected static class CodeDataSourceAutoConfiguration implements ApplicationContextAware {
		
		private static final Log logger = LogFactory.getLog(CodeDataSourceAutoConfiguration.class);
		
		@Autowired
		private ApplicationContext applicationContext;
		
		@Autowired
		private CodeDataSourceProperties codeDataSourceProperties;
		
		@Bean
		@ConditionalOnBean(name=SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME)
		public DBReloadableResourceBundleCodeSource codeSourceForDb() {
			logger.info("===== CodeSource DataSource Enabled: True");
			
			DBReloadableResourceBundleCodeSource codeDataSource = new DBReloadableResourceBundleCodeSource();
			codeDataSource.setDataSource((DataSource) this.applicationContext.getBean(SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME));
			
			if(codeDataSourceProperties.isFallbackToSystemLocale())
				codeDataSource.setFallbackToSystemLocale(codeDataSourceProperties.isFallbackToSystemLocale());
			if(codeDataSourceProperties.getName() != null)	
				codeDataSource.setBeanName(codeDataSourceProperties.getName());
			if(codeDataSourceProperties.getOrder() != Integer.MAX_VALUE)	
				codeDataSource.setOrder(codeDataSourceProperties.getOrder());
			if(codeDataSourceProperties.getCacheSeconds() != 0)
				codeDataSource.setCacheSeconds(codeDataSourceProperties.getCacheSeconds());
			if(codeDataSourceProperties.getQueryCode() != null)
				codeDataSource.setQueryCode(codeDataSourceProperties.getQueryCode());
			
			logger.info("====== Creating CodeSource For ParentMessageSource.");
			return codeDataSource;
		}

		@Override
		public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
			this.applicationContext = applicationContext;
		}

	}
	
	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}
	
}