package nexcore.sprout.spring.boot.autoconfigure.reload;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.jedis.JedisConnection;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.listener.ChannelTopic;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.data.redis.listener.adapter.MessageListenerAdapter;

import nexcore.sprout.foundry.reload.ReloadMulticaster;
import nexcore.sprout.spring.boot.autoconfigure.SproutAutoConfigConst;
import redis.clients.jedis.Jedis;

@Configuration
@ConditionalOnProperty(prefix = SproutAutoConfigConst.PREFIX_SPROUT_RELOAD_MULTICASTER, name = "enabled", havingValue="true")
public class ReloadAutoConfiguration{

	private static final Log logger = LogFactory.getLog(ReloadAutoConfiguration.class);

	@Bean
	public ReloadMulticaster reloadMulticaster(){
 		ReloadMulticaster reloadMulticaster = new ReloadMulticaster();
		logger.info("====== Creating Bean: " + reloadMulticaster.getClass().toString());
		return reloadMulticaster;
	}
	
	@Configuration
	@ConditionalOnClass({ JedisConnection.class, RedisOperations.class, Jedis.class })
	@ConditionalOnProperty(prefix = SproutAutoConfigConst.PREFIX_SPROUT_RELOAD_REDIS, name = "enabled", havingValue="true")
	@AutoConfigureAfter(ReloadMulticaster.class)
	public static class ReloadForRedis implements ApplicationContextAware{
		private static final Log logger = LogFactory.getLog(ReloadForRedis.class);

		@Autowired
		private ApplicationContext applicationContext;
		
		/**
		 * 
		 * ChannelTopic을 이용해 RedisConnectionFactory에서 Message를 수신받아 
		 * 등록된 MessageListenerAdapter로 Message를 Dispatch
		 * 
		 * @param connectionFactory
		 * @param listenerAdapter
		 * @return
		 */
		@Bean
		public RedisMessageListenerContainer container(RedisConnectionFactory connectionFactory,
				MessageListenerAdapter listenerAdapter) {
			RedisMessageListenerContainer container = new RedisMessageListenerContainer();
			container.setConnectionFactory(connectionFactory);
			container.addMessageListener(listenerAdapter, applicationContext.getBean("topic",ChannelTopic.class));

			logger.info("====== Creating Bean: " + container.getClass().toString());
			return container;
		}

		/**
		 * 
		 * Foundy ReloadMulticaster를 Receiver 로 하여 MessageListenerContainer에서 받아온 Message를
		 * Receiver의 SproutAutoConfigConst.REDIS_RELOADMULTICAST_METHOD_NAME로 전송
		 * 
		 * @param receiver
		 * @return
		 */
		@Bean
		public MessageListenerAdapter listenerAdapter(ReloadMulticaster receiver) {
			MessageListenerAdapter listenerAdapter = new MessageListenerAdapter();
			listenerAdapter.setDelegate(receiver);
			listenerAdapter.setDefaultListenerMethod(SproutAutoConfigConst.REDIS_RELOADMULTICAST_METHOD_NAME);
			logger.info("====== Creating Bean: " + listenerAdapter.getClass().toString());
			return listenerAdapter;
		}
		
		/**
		 * 
		 * Redis pub/sub통신 중 Subscribe할 때 Key역할을 하는 Topic설정
		 * Publish와 동일한 Topic을 설정함
		 * 
		 * @return
		 */
		@Bean
		public ChannelTopic topic() {  
		    String appName = applicationContext.getEnvironment().getProperty(SproutAutoConfigConst.SPRING_APPLICATION_NAME_PROPERTY);
		    String topic = SproutAutoConfigConst.REDIS_RELOADMULTICAST_TOPIC;
		    if(appName!= null){
		    	topic+=appName;
		    }
			logger.info("====== Redis Topic : " + topic);
		    return new ChannelTopic(topic);
		}

		@Override
		public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
			this.applicationContext=applicationContext;
		}

	}
}
