package nexcore.sprout.spring.boot.autoconfigure.exception;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactoryUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import nexcore.sprout.foundry.exception.hanlder.ExceptionTracer;
import nexcore.sprout.foundry.exception.hanlder.impl.DefaultExceptionLogTracer;

@Configuration
@EnableConfigurationProperties(SproutExceptionProperties.class)
@AutoConfigureBefore(SproutExceptionHandlerAutoConfiguration.class)
public class SproutExceptionTracerConfiguration implements ApplicationContextAware {
	
	private static final Log logger = LogFactory.getLog(SproutExceptionTracerConfiguration.class);
	
	@Autowired
	private ConfigurableApplicationContext applicationContext;
	
	@Autowired
	private SproutExceptionProperties exceptionProperties;
	
	/**
	 * ExceptionHandler에서 ExceptionTracer는 Multi로 설정가능
	 * 따라서, ExceptionTracer가 아닌 DefaultExceptionLogTracer가 없을 경우에 Bean 생성
	 * @return
	 */
	@Bean
	@ConditionalOnMissingBean
	public DefaultExceptionLogTracer defaultExceptionLogTracer(){
		DefaultExceptionLogTracer defaultExceptionLogTracer = new DefaultExceptionLogTracer();
		
		if(exceptionProperties.isPrintStackTrace())
			defaultExceptionLogTracer.setPrintStackTrace(exceptionProperties.isPrintStackTrace());
				
		logger.info("====== Creating Bean: " + defaultExceptionLogTracer.getClass().toString());
		if(logger.isDebugEnabled()) {
			logger.debug("====== Properties for Created Bean");
			logger.debug("====== PrintStackTrace: " + exceptionProperties.isPrintStackTrace());
		}
		
		return defaultExceptionLogTracer;
	}
	
	public List<ExceptionTracer> findExceptionTracer() {
		List<ExceptionTracer> exceptionTracerList = new ArrayList<ExceptionTracer>();
		
		if (this.applicationContext == null) {
			logger.info("ExceptionLogTracer List is null.");
			return Collections.emptyList();
		}
		
		exceptionTracerList.addAll(getBeans(ExceptionTracer.class));
		
		if(logger.isDebugEnabled()) {
			logger.debug("====== Find Total ExceptionTracer: " + exceptionTracerList.size());
			for (ExceptionTracer exceptionTracer : exceptionTracerList) {
				logger.debug("====== Find ExceptionTracer: " + exceptionTracer.getClass().toString());
			}
		}
		
		return exceptionTracerList;
	}
	
	private <T> Collection<T> getBeans(Class<T> type) {
		return BeanFactoryUtils.beansOfTypeIncludingAncestors(this.applicationContext.getBeanFactory(), type).values();
	}
	
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		if (applicationContext instanceof ConfigurableApplicationContext) {
			this.applicationContext = (ConfigurableApplicationContext) applicationContext;
		}
	}

}
