package nexcore.sprout.spring.boot.autoconfigure.exception;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactoryUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.PropertiesFactoryBean;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import nexcore.sprout.foundry.exception.hanlder.ExceptionCodeResolver;
import nexcore.sprout.foundry.exception.hanlder.impl.DefaultExceptionCodeResolver;
import nexcore.sprout.foundry.exception.hanlder.impl.PropertyExceptionCodeResolver;
import nexcore.sprout.spring.boot.autoconfigure.SproutAutoConfigConst;

@Configuration
@EnableConfigurationProperties(SproutExceptionProperties.class)
@AutoConfigureBefore(SproutExceptionHandlerAutoConfiguration.class)
public class SproutExceptionCodeResolverConfiguration implements ApplicationContextAware {
	
	private static final Log logger = LogFactory.getLog(SproutExceptionCodeResolverConfiguration.class);
	
	@Autowired
	private ConfigurableApplicationContext applicationContext;
	
	@Autowired
	private SproutExceptionProperties exceptionProperties;

	/**
	 * ExceptionHandler에서 ExceptionCodeResolver는 Multi로 설정가능
	 * 따라서, ExceptionCodeResolver가 아닌 DefaultExceptionCodeResolver가 없을 경우에 Bean 생성
	 * @return
	 */
	@Bean
	@ConditionalOnMissingBean
	public DefaultExceptionCodeResolver defaultExceptionCodeResolver(){
		DefaultExceptionCodeResolver defaultExceptionCodeResolver = new DefaultExceptionCodeResolver();
		logger.info("====== Creating Bean: " + defaultExceptionCodeResolver.getClass().toString());
		return defaultExceptionCodeResolver;
	}
	
	/**
	 * ExceptionHandler에서 ExceptionCodeResolver는 Multi로 설정가능
	 * 따라서, ExceptionCodeResolver가 아닌 PropertyExceptionCodeResolver가 없을 경우에 Bean 생성 
	 * @return
	 * @throws IOException
	 */
	@Bean
	@ConditionalOnProperty(prefix = SproutAutoConfigConst.PREFIX_SPROUT_EXCEPTION, name = "location")
	public PropertiesFactoryBean exceptionPropertiesFactoryBean() throws IOException{
		PropertiesFactoryBean errorProperties = new PropertiesFactoryBean();
		errorProperties.setLocation(exceptionProperties.getLocation());
		
		if(exceptionProperties.isIgnoreResourceNotFound())
			errorProperties.setIgnoreResourceNotFound(exceptionProperties.isIgnoreResourceNotFound());
		if(exceptionProperties.isLocalOverride())
			errorProperties.setLocalOverride(exceptionProperties.isLocalOverride());
		
		logger.info("====== Creating Property Bean: " + errorProperties.getClass().toString());
		if(logger.isDebugEnabled()) {
			logger.debug("====== Properties for Created Bean");
			logger.debug("====== IgnoreResourceNotFounde: " + exceptionProperties.isIgnoreResourceNotFound());
			logger.debug("====== LocalOverride: " + exceptionProperties.isLocalOverride());
			logger.debug("====== Location: " + exceptionProperties.getLocation());
		}
		return errorProperties;
	}
	
	/**
	 * PropertiesFactoryBean에서 생성된 properties를 사용해 propertyExceptionCodeResolver Bean생성
	 * @return
	 * @throws BeansException
	 * @throws IOException
	 */
	@Bean
	@ConditionalOnBean(name = SproutAutoConfigConst.EXCIPTION_PROPERTIES_FACTORY_BEAN)
	public PropertyExceptionCodeResolver propertyExceptionCodeResolver() throws BeansException, IOException{
		PropertyExceptionCodeResolver propertyExceptionCodeResolver = new PropertyExceptionCodeResolver();
		propertyExceptionCodeResolver.setProperties(this.applicationContext.getBean(PropertiesFactoryBean.class,SproutAutoConfigConst.EXCIPTION_PROPERTIES_FACTORY_BEAN).getObject());

		return propertyExceptionCodeResolver;
	}
	
	public List<ExceptionCodeResolver> findExceptionCodeReoslvers() {
		List<ExceptionCodeResolver> exceptionCodeResolverList = new ArrayList<ExceptionCodeResolver>();
		
		if (this.applicationContext == null) {
			logger.info("ExceptionCodeResolver List is null.");
			return Collections.emptyList();
		}
		
		exceptionCodeResolverList.addAll(getBeans(ExceptionCodeResolver.class));
		
		if(logger.isDebugEnabled()) {
			logger.debug("====== Find Total ExceptionCodeResolver: " + exceptionCodeResolverList.size());
			for (ExceptionCodeResolver exceptionCodeResolver : exceptionCodeResolverList) {
				logger.debug("====== Find ExceptionCodeResolver: " + exceptionCodeResolver.getClass().toString());
			}
		}
	
		return exceptionCodeResolverList;
	}
	
	private <T> Collection<T> getBeans(Class<T> type) {
		return BeanFactoryUtils.beansOfTypeIncludingAncestors(this.applicationContext.getBeanFactory(), type).values();
	}

	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		if (applicationContext instanceof ConfigurableApplicationContext) {
			this.applicationContext = (ConfigurableApplicationContext) applicationContext;
		}
	}
	
}
