package nexcore.sprout.spring.boot.autoconfigure.webmvc;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.embedded.FilterRegistrationBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.OrderComparator;
import org.springframework.core.Ordered;
import org.springframework.web.accept.ContentNegotiationManager;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.View;
import org.springframework.web.servlet.ViewResolver;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.ViewControllerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import org.springframework.web.servlet.i18n.LocaleChangeInterceptor;
import org.springframework.web.servlet.i18n.SessionLocaleResolver;
import org.springframework.web.servlet.view.BeanNameViewResolver;
import org.springframework.web.servlet.view.ContentNegotiatingViewResolver;
import org.springframework.web.servlet.view.InternalResourceViewResolver;
import org.springframework.web.servlet.view.json.MappingJackson2JsonView;

import nexcore.sprout.foundry.webmvc.filter.GuidGenerateServletFilter;
import nexcore.sprout.foundry.webmvc.filter.MDCInsertingServletFilter;
import nexcore.sprout.foundry.webmvc.filter.RequestContextFilter;
import nexcore.sprout.spring.boot.autoconfigure.SproutAutoConfigConst;

@Configuration
@ConditionalOnWebApplication
@EnableConfigurationProperties(SproutWebMvcProperties.class)
public class SproutWebMvcAutoConfiguration extends WebMvcConfigurerAdapter implements ApplicationContextAware {
	
	private static final Log logger = LogFactory.getLog(SproutWebMvcAutoConfiguration.class);

	@Autowired
	private ApplicationContext applicationContext;
	
	@Autowired
	private SproutWebMvcProperties sproutWebMvcProperties;
	
	@Bean
	@ConditionalOnMissingBean
	public FilterRegistrationBean requestContextFilter() {
		FilterRegistrationBean registration = new FilterRegistrationBean();
		registration.setFilter(new RequestContextFilter());
		registration.setOrder(Ordered.HIGHEST_PRECEDENCE);
		registration.setName("requestContextFilter");
		return registration;
	}
	
	@Bean
	@ConditionalOnMissingBean
	public FilterRegistrationBean guidGenerateServletFilter() {
		FilterRegistrationBean registration = new FilterRegistrationBean();
		registration.setFilter(new GuidGenerateServletFilter());
		registration.setOrder(Ordered.LOWEST_PRECEDENCE - 20);
		registration.setName(SproutAutoConfigConst.GUID_GENERATE_SERVLET_FILTER);
		return registration;
	}
	
	@Bean
	public FilterRegistrationBean MDCInsertingServletFilter() {
		FilterRegistrationBean registration = new FilterRegistrationBean();
		registration.setFilter(new MDCInsertingServletFilter());
		registration.setOrder(Ordered.LOWEST_PRECEDENCE - 10);
		registration.setName(SproutAutoConfigConst.MDC_INSERTING_SERVLET_FILTER);
		return registration;
	}
	
	@Bean
	@ConditionalOnMissingBean
    public LocaleResolver localeResolver() {
        SessionLocaleResolver sessionLocaleResolver = new SessionLocaleResolver();
        
        if(sproutWebMvcProperties.getDefaultLocale() != null)
        	sessionLocaleResolver.setDefaultLocale(sproutWebMvcProperties.getDefaultLocale());
        
        logger.info("====== Creating Locale Resolver: " + sessionLocaleResolver.getClass().toString());
        logger.info("====== Default Locale: " + sproutWebMvcProperties.getDefaultLocale().toString());
        return sessionLocaleResolver;
    }
	
	@Bean
	@ConditionalOnMissingBean
    public LocaleChangeInterceptor localeChangeInterceptor() {
        LocaleChangeInterceptor localeChangeInterceptor = new LocaleChangeInterceptor();
        
        if(sproutWebMvcProperties.getLocaleParamName() != null)
        	localeChangeInterceptor.setParamName(sproutWebMvcProperties.getLocaleParamName());
        
        logger.info("====== Creating LocaleChangeInterceptor: " + localeChangeInterceptor.getClass().toString());
        logger.info("====== LocaleChangeInterceptor for ParamName: " + sproutWebMvcProperties.getLocaleParamName().toString());
        return localeChangeInterceptor;
    }
	
	@Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(this.applicationContext.getBean(LocaleChangeInterceptor.class));
        logger.info("====== Add Interceptor localeChangeInterceptor");
    }
	
	@Override
	public void addViewControllers(ViewControllerRegistry registry) {
		List<nexcore.sprout.spring.boot.autoconfigure.webmvc.SproutWebMvcProperties.View> viewList = sproutWebMvcProperties.getView();
		try{
			for (nexcore.sprout.spring.boot.autoconfigure.webmvc.SproutWebMvcProperties.View view : viewList) {
				registry.addViewController(view.getViewController())
						.setViewName(view.getViewName());
			}
		} catch (Exception e) {
			logger.info("No View Controller");
		}
	}	
	
	@Bean
	@ConditionalOnBean(View.class)
	public BeanNameViewResolver beanNameViewResolver() {
		BeanNameViewResolver resolver = new BeanNameViewResolver();
		resolver.setOrder(Ordered.HIGHEST_PRECEDENCE);
		logger.info("====== Creating Resolver: " + resolver.getClass().toString());
		return resolver;
	}
	
	@Bean(name=SproutAutoConfigConst.VIEW_RESOLVER)
	@ConditionalOnBean(ViewResolver.class)
	public ContentNegotiatingViewResolver viewResolver(BeanFactory beanFactory) {
		ContentNegotiatingViewResolver resolver = new ContentNegotiatingViewResolver();
		resolver.setContentNegotiationManager(beanFactory.getBean(ContentNegotiationManager.class));
			
		Map<String, InternalResourceViewResolver> viewResolvers = this.applicationContext.getBeansOfType(InternalResourceViewResolver.class);
		List<ViewResolver> viewResolverList = new ArrayList<ViewResolver>(viewResolvers.values());
		OrderComparator.sort(viewResolverList);
		resolver.setViewResolvers(viewResolverList);
		if(logger.isDebugEnabled()) {
            logger.debug("====== ViewResolver List in ContentNegotiatingViewResolver ======");
            for (ViewResolver viewResolver : viewResolverList) {
                logger.debug("ViewResolver: " +  viewResolver.getClass());
            }
            logger.debug("=================================================================");
        }
		
		List<View> defaultViewList = new ArrayList<View>();
		defaultViewList.add(new MappingJackson2JsonView());
		resolver.setDefaultViews(defaultViewList);
		if(logger.isDebugEnabled()) {
            logger.debug("====== DefaultView List in ContentNegotiatingViewResolver ======");
            for (View defaultView : defaultViewList) {
                logger.debug("Default View: " +  defaultView.getClass());
            }
            logger.debug("================================================================");
        }
		
		resolver.setOrder(Ordered.HIGHEST_PRECEDENCE + 10);
		return resolver;
	}

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		if (applicationContext instanceof ConfigurableApplicationContext) {
			this.applicationContext = (ConfigurableApplicationContext) applicationContext;
		}
	}
	
}
