package nexcore.sprout.spring.boot.autoconfigure.tiles;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.tiles.startup.TilesInitializer;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.OrderComparator;
import org.springframework.core.Ordered;
import org.springframework.util.StringUtils;
import org.springframework.web.accept.ContentNegotiationManager;
import org.springframework.web.servlet.View;
import org.springframework.web.servlet.ViewResolver;
import org.springframework.web.servlet.view.ContentNegotiatingViewResolver;
import org.springframework.web.servlet.view.UrlBasedViewResolver;
import org.springframework.web.servlet.view.json.MappingJackson2JsonView;
import org.springframework.web.servlet.view.tiles3.SpringBeanPreparerFactory;
import org.springframework.web.servlet.view.tiles3.TilesConfigurer;
import org.springframework.web.servlet.view.tiles3.TilesView;

import nexcore.sprout.spring.boot.autoconfigure.SproutAutoConfigConst;
import nexcore.sprout.spring.boot.autoconfigure.webmvc.SproutWebMvcAutoConfiguration;
import nexcore.sprout.spring.boot.autoconfigure.webmvc.SproutWebMvcProperties;

@Configuration
@ConditionalOnClass(TilesInitializer.class)
@AutoConfigureAfter(SproutWebMvcAutoConfiguration.class)
@EnableConfigurationProperties({SproutTilesProperties.class,SproutWebMvcProperties.class})
public class SproutTilesAutoConfiguration implements ApplicationContextAware {

	private static final Log logger = LogFactory.getLog(SproutTilesAutoConfiguration.class);

	private ApplicationContext applicationContext;
	
	@Autowired
	private SproutWebMvcProperties sproutWebMvcProperties;
	
	@Autowired
	private SproutTilesProperties sproutTilesProperties;
	
    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnProperty(prefix = SproutAutoConfigConst.PREFIX_SPROUT_TILES , value = "path")
    public TilesConfigurer tilesConfigurer() {
        TilesConfigurer configurer = new TilesConfigurer();
        configurer.setDefinitions(StringUtils.commaDelimitedListToStringArray(StringUtils.trimAllWhitespace(sproutTilesProperties.getPath())));
        logger.info("====== Set the Tiles Configuration File Location.");

        if(SpringBeanPreparerFactory.class != null) {
        	logger.info("====== Set the SpringBeanPreparerFactory ");
        	configurer.setPreparerFactoryClass(SpringBeanPreparerFactory.class);
        }
        return configurer;
    }      
    
    @Bean
	public UrlBasedViewResolver doView(){
		UrlBasedViewResolver doView = new UrlBasedViewResolver();
		doView.setViewClass(TilesView.class);
		doView.setPrefix(sproutWebMvcProperties.getDoPrefix());
		doView.setSuffix(sproutWebMvcProperties.getDoSuffix());
		doView.setOrder(Ordered.HIGHEST_PRECEDENCE);
		return doView;
	}
	
    @Bean(name=SproutAutoConfigConst.FRAG_VIEW)
	@ConditionalOnMissingBean(name=SproutAutoConfigConst.FRAG_VIEW)
	public UrlBasedViewResolver fragView(){
		UrlBasedViewResolver fragView = new UrlBasedViewResolver();
		fragView.setViewClass(TilesView.class);
		fragView.setPrefix(sproutWebMvcProperties.getFragPrefix());
		fragView.setSuffix(sproutWebMvcProperties.getFragSuffix());
		fragView.setContentType(SproutAutoConfigConst.APPLICATION_PLAIN);
		fragView.setOrder(Ordered.HIGHEST_PRECEDENCE + 10);
		return fragView;
	}
    
    @Bean(name=SproutAutoConfigConst.VIEW_RESOLVER)
	@ConditionalOnBean(ViewResolver.class)
	public ContentNegotiatingViewResolver viewResolver(BeanFactory beanFactory) {
		ContentNegotiatingViewResolver resolver = new ContentNegotiatingViewResolver();
		resolver.setContentNegotiationManager(beanFactory.getBean(ContentNegotiationManager.class));
			
		Map<String, UrlBasedViewResolver> viewResolvers = this.applicationContext.getBeansOfType(UrlBasedViewResolver.class);
		List<ViewResolver> viewResolverList = new ArrayList<ViewResolver>(viewResolvers.values());
		OrderComparator.sort(viewResolverList);
		resolver.setViewResolvers(viewResolverList);
		if(logger.isDebugEnabled()) {
            logger.debug("====== ViewResolver List in ContentNegotiatingViewResolver ======");
            for (ViewResolver viewResolver : viewResolverList) {
                logger.debug("ViewResolver: " +  viewResolver.getClass());
            }
            logger.debug("=================================================================");
        }
		
		List<View> defaultViewList = new ArrayList<View>();
		defaultViewList.add(new MappingJackson2JsonView());
		resolver.setDefaultViews(defaultViewList);
		if(logger.isDebugEnabled()) {
            logger.debug("====== DefaultView List in ContentNegotiatingViewResolver ======");
            for (View defaultView : defaultViewList) {
                logger.debug("Default View: " +  defaultView.getClass());
            }
            logger.debug("================================================================");
        }
		
		resolver.setOrder(Ordered.HIGHEST_PRECEDENCE);
		return resolver;
	}

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}

}
