package nexcore.sprout.spring.boot.autoconfigure.resourcebundle;

import javax.sql.DataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import nexcore.sprout.foundry.resourcebundle.message.DBReloadableResourceBundleMessageSource;
import nexcore.sprout.foundry.resourcebundle.message.ReloadableResourceBundleMessageSource;
import nexcore.sprout.spring.boot.autoconfigure.SproutAutoConfigConst;
import nexcore.sprout.spring.boot.autoconfigure.jdbc.SproutDataSourceAutoConfiguration;
import nexcore.sprout.spring.boot.autoconfigure.resourcebundle.MessageSourceProperties.MessageDataSourceProperties;

@Configuration
@EnableConfigurationProperties(MessageSourceProperties.class)
@AutoConfigureBefore(MessageSourceAccessorAutoConfiguration.class)
@AutoConfigureAfter(SproutDataSourceAutoConfiguration.class)
@ConditionalOnProperty(prefix = SproutAutoConfigConst.PREFIX_SPROUT_MESSAGE_RESOURCE, name = "basename")
public class MessageSourceAutoConfiguration implements ApplicationContextAware {
	
	private static final Log logger = LogFactory.getLog(MessageSourceAutoConfiguration.class);
	
	@Autowired
	private ApplicationContext applicationContext;
	
	@Autowired
	private MessageSourceProperties messageSourceProperties;
	
	@Bean
	public ReloadableResourceBundleMessageSource messageSource() {
		ReloadableResourceBundleMessageSource messageSource = new ReloadableResourceBundleMessageSource();
		messageSource.setBasenames(StringUtils.commaDelimitedListToStringArray(StringUtils.trimAllWhitespace(messageSourceProperties.getBasename())));
		
		if(messageSourceProperties.getDefaultEncoding() != null)
			messageSource.setDefaultEncoding(messageSourceProperties.getDefaultEncoding());
		if(messageSourceProperties.getDefaultFilePostFix() != null)
			messageSource.setDefaultFilePostFix(messageSourceProperties.getDefaultFilePostFix());
		if(messageSourceProperties.isUseCustomDefaultFile())
			messageSource.setUseCustomDefaultFile(messageSourceProperties.isUseCustomDefaultFile());
		if(messageSourceProperties.isUseAutoReloading())
			messageSource.setUseAutoReloading(messageSourceProperties.isUseAutoReloading());
		if(messageSourceProperties.getName() != null)
			messageSource.setBeanName(messageSourceProperties.getName());
		
		try {
			DBReloadableResourceBundleMessageSource messageSourceForDb = this.applicationContext.getBean(DBReloadableResourceBundleMessageSource.class);
			messageSource.setParentMessageSource(messageSourceForDb);
			logger.info("====== MessageSource For ParentMessageSource is created.");
		} catch (NoSuchBeanDefinitionException ex) {
			logger.info("====== MessageSource For ParentMessageSource is not created.");
		}
		
		logger.info("====== Creating ReloadableResourceBundleMessageSource Bean: " + messageSource.getClass().toString());
		return messageSource;
	}
	
	@Configuration	
	@EnableConfigurationProperties(MessageDataSourceProperties.class)
	@AutoConfigureBefore(MessageSourceAutoConfiguration.class)
	@AutoConfigureAfter(SproutDataSourceAutoConfiguration.class)
	@ConditionalOnProperty(prefix = SproutAutoConfigConst.PREFIX_SPROUT_MESSAGE_DATASOURCE, name = "enabled", havingValue = "true")
	protected static class SproutMessageDataSourceAutoConfiguration implements ApplicationContextAware{
		
		private static final Log logger = LogFactory.getLog(SproutMessageDataSourceAutoConfiguration.class);
		
		@Autowired
		private ApplicationContext applicationContext;

		@Autowired
		private MessageDataSourceProperties messageDataSourceProperties;
		
		@Bean
		@ConditionalOnBean(name=SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME)
		public DBReloadableResourceBundleMessageSource messageSourceForDb(){
			logger.info("===== MessageSource DataSource Enabled: True");
			
			DBReloadableResourceBundleMessageSource messageDatasource = new DBReloadableResourceBundleMessageSource();
			messageDatasource.setDataSource((DataSource) this.applicationContext.getBean(SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME));
			
			if(messageDataSourceProperties.isFallbackToSystemLocale())
				messageDatasource.setFallbackToSystemLocale(messageDataSourceProperties.isFallbackToSystemLocale());
			if(messageDataSourceProperties.getCacheSeconds() != 0)
				messageDatasource.setCacheSeconds(messageDataSourceProperties.getCacheSeconds());
			if(messageDataSourceProperties.getQueryCode() != null)
				messageDatasource.setQueryCode(messageDataSourceProperties.getQueryCode());
			if(messageDataSourceProperties.getName() != null)
				messageDatasource.setBeanName(messageDataSourceProperties.getName());
			
			logger.info("====== Creating MessageSource For ParentMessageSource.");
			return messageDatasource;
		}
		
		@Override
		public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
			this.applicationContext = applicationContext;
		}

	}

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}

}