package nexcore.sprout.spring.boot.autoconfigure.log.monitor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.springframework.beans.factory.BeanFactoryUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import nexcore.sprout.log.monitor.PeriodLogTimer;
import nexcore.sprout.log.monitor.SproutHttpSessionCountListener;
import nexcore.sprout.log.monitor.gatherer.IPeriodicalLogGatherer;
import nexcore.sprout.log.monitor.gatherer.SessionCountGatherer;
import nexcore.sprout.log.monitor.gatherer.SystemDataGatherer;
import nexcore.sprout.log.monitor.processor.ILogDataProcessor;
import nexcore.sprout.log.monitor.processor.SystemStatusLogDataProcessor;
import nexcore.sprout.log.monitor.writer.logstash.PeriodLogstashLogWriter;
import nexcore.sprout.spring.boot.autoconfigure.SproutAutoConfigConst;

@Configuration
@EnableConfigurationProperties({ LogMonitorProperties.class})
@ConditionalOnProperty(prefix=SproutAutoConfigConst.PREFIX_SPROUT_LOG_MONITOR, name="enabled", havingValue="true")
public class PeriodLogWriterConfiguration {

	@Autowired
	private ConfigurableApplicationContext applicationContext;

	@Autowired
	private LogMonitorProperties logMonitorProperties;
	
	@Bean
	public PeriodLogTimer periodLogTimer() {
		PeriodLogTimer periodLogTimer = new PeriodLogTimer(logMonitorProperties.getWriteLogInterval(), periodLogstashLogWriter());
		periodLogTimer.setUseYN(true);
		return periodLogTimer;
	}
	
	@Bean
	public PeriodLogstashLogWriter periodLogstashLogWriter() {
		PeriodLogstashLogWriter periodLogstashLogWriter = new PeriodLogstashLogWriter();
		periodLogstashLogWriter.setLogDataProcessors(findILogDataProcessor());
		periodLogstashLogWriter.setLogstashLoggerName(logMonitorProperties.getPeriodLoggerName());
		return periodLogstashLogWriter;
	}
	
	@Bean
	@ConditionalOnMissingBean(name={SproutAutoConfigConst.SYSTEM_DATA_GATHERER_NAME})
	public IPeriodicalLogGatherer systemDataGatherer() {
		return new SystemDataGatherer();
	}
	
	@Bean
	@ConditionalOnMissingBean(name={SproutAutoConfigConst.SESSION_COUNT_GATHERER_NAME})
	public IPeriodicalLogGatherer sessionCountGatherer() {
		SessionCountGatherer sessionCountGatherer = new SessionCountGatherer();
		sessionCountGatherer.setListener(sessionListener());
		return sessionCountGatherer;
	}
	
	@Bean
	@ConditionalOnMissingBean(name={SproutAutoConfigConst.SYSTEM_STATUS_LOG_DATA_PROCESSOR_NAME})
	public ILogDataProcessor systemStatusLogDataProcessor() {
		SystemStatusLogDataProcessor systemStatusLogDataProcessor = new SystemStatusLogDataProcessor();
		systemStatusLogDataProcessor.setLogGatherers(findILogGatherer());
		systemStatusLogDataProcessor.setServiceGroupName(logMonitorProperties.getPeriodLoggerName());
		return systemStatusLogDataProcessor;
	}
	
	private <T> Collection<T> getBeans(Class<T> type) {
		return BeanFactoryUtils.beansOfTypeIncludingAncestors(this.applicationContext.getBeanFactory(), type).values();
	}
	
	@Bean
	@ConditionalOnMissingBean(name={SproutAutoConfigConst.SESSION_LISTENER_NAME})
	public SproutHttpSessionCountListener sessionListener() {
		return new SproutHttpSessionCountListener();
	}
	
	public List<IPeriodicalLogGatherer> findILogGatherer() {
		List<IPeriodicalLogGatherer> iLogGathererList = new ArrayList<IPeriodicalLogGatherer>();

		if (this.applicationContext == null) {
			return Collections.emptyList();
		}

		for (IPeriodicalLogGatherer iLogGatherer : getBeans(IPeriodicalLogGatherer.class)) {
			iLogGathererList.add(iLogGatherer);
		}
		return iLogGathererList;
	}
	
	public List<ILogDataProcessor> findILogDataProcessor() {
		List<ILogDataProcessor> iLogDataProcessorList = new ArrayList<ILogDataProcessor>();

		if (this.applicationContext == null) {
			return Collections.emptyList();
		}

		for (ILogDataProcessor iLogDataProcessor : getBeans(ILogDataProcessor.class)) {
			iLogDataProcessorList.add(iLogDataProcessor);
		}
		return iLogDataProcessorList;
	}
	
	@EventListener
	@Order(Ordered.LOWEST_PRECEDENCE)
	public void onClosedContext(ContextClosedEvent event) {
		periodLogTimer().stopTimer();
	}
}
