package nexcore.sprout.spring.boot.autoconfigure.jdbc;

import javax.sql.DataSource;
import javax.sql.XADataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.autoconfigure.jdbc.DataSourceBuilder;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;
import org.springframework.core.Ordered;
import org.springframework.jdbc.datasource.embedded.EmbeddedDatabaseType;
import org.springframework.jdbc.datasource.lookup.JndiDataSourceLookup;
import org.springframework.jmx.export.MBeanExporter;
import org.springframework.jmx.support.JmxUtils;

import net.sf.log4jdbc.Log4jdbcProxyDataSource;
import net.sf.log4jdbc.SpyLogDelegator;
import nexcore.sprout.foundry.log.sqllog.Log4JdbcCustomFormatter;
import nexcore.sprout.spring.boot.autoconfigure.SproutAutoConfigConst;

/**
 * DataSourceAutoConfiguration 설정 전에 실행됨
 * 
 */
@Configuration
@AutoConfigureBefore(DataSourceAutoConfiguration.class)
@ConditionalOnClass({ DataSource.class, EmbeddedDatabaseType.class })
@EnableConfigurationProperties({ DataSourceProperties.class, SproutDataSourceProperties.class, SproutSqlLogProperties.class })
@ConditionalOnProperty(prefix=SproutAutoConfigConst.PREFIX_SPROUT_DATASOURCE,name="useConfig",matchIfMissing=true )
public class SproutDataSourceAutoConfiguration {

	private static final Log logger = LogFactory.getLog(SproutDataSourceAutoConfiguration.class);

	@Autowired
	private DataSourceProperties properties;

	@Autowired
	private SproutDataSourceProperties sproutDataSourceProperties;

	@Autowired
	private SproutSqlLogProperties sproutSqlLogProperties;
	
	@Autowired
	private ApplicationContext context;
	
	@Autowired
	private BeanFactory beanFactory;
	
	@Bean(name = SproutAutoConfigConst.DEFAULT_DATASOURCE_NAME)
	@ConditionalOnMissingBean({ DataSource.class, XADataSource.class })
	@ConditionalOnProperty(prefix = DataSourceProperties.PREFIX, name = "jndi-name")
	@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE   )
	public DataSource jndiDataSource() {
		return  wrappingLog4jdbcDataSource(createJndiDataSource());
	}
	
	@Bean(name = SproutAutoConfigConst.DEFAULT_DATASOURCE_NAME)
	@ConditionalOnMissingBean({ DataSource.class, XADataSource.class })
	@ConfigurationProperties(prefix = DataSourceProperties.PREFIX)
	@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE+1)
	public DataSource dataSource() {
		return  wrappingLog4jdbcDataSource(createDataSource(this.properties));
	}  
	
	@ConditionalOnMissingBean(name = SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME)
	@ConditionalOnProperty(prefix = DataSourceProperties.PREFIX, name = "jndi-name")
	@Bean(name = SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME)
	@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE + 2)
	public Object dataSourceForSproutJndi() {
		logger.info("create datasourceForSprout");
		return createJndiDataSource();
	}
	
	@ConditionalOnMissingBean(name = SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME)
	@Bean(name = SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME)
	@AutoConfigureOrder(Ordered.HIGHEST_PRECEDENCE + 3)
	public Object dataSourceForSprout(DataSource dataSource) {
		logger.info("create datasourceForSprout");
		return createDataSource(this.properties);
	}

	@Bean
	@ConditionalOnMissingBean
	public Log4JdbcCustomFormatter sqlFormatter() {
		Log4JdbcCustomFormatter sqlFormatter = new Log4JdbcCustomFormatter();
		sqlFormatter.setSqlPrefix(sproutSqlLogProperties.getSqlLogPrefix());
		sqlFormatter.setLoggingCallMethod(sproutSqlLogProperties.isShowCallClass());
		sqlFormatter.setLineLength(sproutSqlLogProperties.getLineLength());
		return sqlFormatter;
	}
	
	private static DataSource createDataSource(DataSourceProperties properties ) {
		DataSourceBuilder factory = DataSourceBuilder.create( properties.getClassLoader())
				.driverClassName( properties.getDriverClassName()).url( properties.getUrl())
				.username( properties.getUsername()).password( properties.getPassword());
		if ( properties.getType() != null) {
			factory.type( properties.getType());
		}
		return factory.build();
	}

	private DataSource createJndiDataSource() {
		JndiDataSourceLookup dataSourceLookup = new JndiDataSourceLookup();
		DataSource dataSource = dataSourceLookup.getDataSource(properties.getJndiName());
		excludeMBeanIfNecessary(dataSource, SproutAutoConfigConst.DEFAULT_DATASOURCE_NAME);
		return dataSource;
	}

	private void excludeMBeanIfNecessary(Object candidate, String beanName) {
		try {
			MBeanExporter mbeanExporter = this.context.getBean(MBeanExporter.class);
			if (JmxUtils.isMBean(candidate.getClass())) {
				mbeanExporter.addExcludedBean(beanName);
			}
		} catch (NoSuchBeanDefinitionException ex) {
		}
	}
	
	private DataSource wrappingLog4jdbcDataSource(DataSource dataSource ){
		if ( sproutSqlLogProperties.isUse()) {
			Log4jdbcProxyDataSource dataSourceForSrpout = new Log4jdbcProxyDataSource(dataSource);
			SpyLogDelegator sl=beanFactory.getBean(SpyLogDelegator.class);
			dataSourceForSrpout.setLogFormatter(sl);
			return dataSourceForSrpout;
		}else{
			return dataSource;
		}
	}
 
	@Bean
	@ConditionalOnMissingClass
	public SproutDataSourceInitializer sproutDataSourceInitializer() {
		return new SproutDataSourceInitializer();
	}
	
	/**
	 * Test 일 경우 WAS에 등록된 DataSource 를 look up 하지 않고  
	 * DBCP 를 이용해 DataSource 를 생성함 
	 * */
	@Configuration
	@Profile(value=SproutAutoConfigConst.ACTIVE_PROFILE_NAME_TEST)
	protected static class SproutDataSourceTestAutoConfiguration  {
		@Autowired
		private DataSourceProperties properties;
		
		@Bean(name = SproutAutoConfigConst.DEFAULT_DATASOURCE_NAME)
		@ConditionalOnMissingBean({ DataSource.class, XADataSource.class })
		@ConfigurationProperties(prefix = DataSourceProperties.PREFIX)
		public DataSource dataSource() {
			return createDataSource(properties);
		}
		
		@ConditionalOnMissingBean(name = SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME)
		@Bean(name = SproutAutoConfigConst.DATASOURCE_FOR_SPROUT_NAME)
		public Object dataSourceForSprout(DataSource dataSource) {
			return createDataSource(properties);
		}
	}
	
}
