package nexcore.sprout.spring.boot.autoconfigure.exception;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactoryUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import nexcore.sprout.foundry.exception.hanlder.ExceptionTranslator;
import nexcore.sprout.foundry.exception.hanlder.web.ModelAndViewExceptionTranslator;

@Configuration
@EnableConfigurationProperties(SproutExceptionProperties.class)
@AutoConfigureBefore(SproutExceptionHandlerAutoConfiguration.class)
public class SproutExceptionTranslatorConfiguration implements ApplicationContextAware {
	
	private static final Log logger = LogFactory.getLog(SproutExceptionTranslatorConfiguration.class);
	
	private ConfigurableApplicationContext applicationContext;
	
	@Autowired
	private SproutExceptionProperties exceptionProperties;
	
	/**
	 * ExceptionHandler에서 ExceptionTraslator는 Multi로 설정가능
	 * 따라서, ExceptionTraslator가 아닌 ModelAndViewExceptionTranslator가 없을 경우에 Bean 생성
	 * @return
	 */
	@Bean
	@ConditionalOnMissingBean
	public ModelAndViewExceptionTranslator MVCExceptionTranslator(){
		ModelAndViewExceptionTranslator mvcExceptionTranslator = new ModelAndViewExceptionTranslator();
		
		if(exceptionProperties.getDefaultViewName() != null)
			mvcExceptionTranslator.setDefaultViewName(exceptionProperties.getDefaultViewName());
		if(exceptionProperties.isSupportOldVersion())
			mvcExceptionTranslator.setSupportOldVersion(exceptionProperties.isSupportOldVersion());
		
		logger.info("====== Creating Bean: " + mvcExceptionTranslator.getClass().toString());
		if(logger.isDebugEnabled()) {
			logger.debug("====== Properties for Created Bean");
			logger.debug("====== DefaultViewName: " + exceptionProperties.getDefaultViewName());
		}
		
		return mvcExceptionTranslator;
	}
	
	public List<ExceptionTranslator> findExceptionTranslator() {
		List<ExceptionTranslator> exceptionTranslatorList = new ArrayList<ExceptionTranslator>();
		
		if (this.applicationContext == null) {
			logger.info("ExceptionTranslator List is null.");
			return Collections.emptyList();
		}
		
		exceptionTranslatorList.addAll(getBeans(ExceptionTranslator.class));
		
		if(logger.isDebugEnabled()) {
			logger.debug("====== Find Total ExceptionTranslator: " + exceptionTranslatorList.size());
			for (ExceptionTranslator exceptionTranslator : exceptionTranslatorList) {
				logger.debug("====== Find ExceptionTranslator: " + exceptionTranslator.getClass().toString());
			}
		}
		
		return exceptionTranslatorList;
	}
	
	private <T> Collection<T> getBeans(Class<T> type) {
		return BeanFactoryUtils.beansOfTypeIncludingAncestors(this.applicationContext.getBeanFactory(), type).values();
	}
	
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		if (applicationContext instanceof ConfigurableApplicationContext) {
			this.applicationContext = (ConfigurableApplicationContext) applicationContext;
		}
	}

}
