package qrscanner

import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import kmpImagePicker.AlertMessageDialog
import kmpImagePicker.utils.PermissionCallback
import kmpImagePicker.utils.PermissionStatus
import kmpImagePicker.utils.PermissionType
import kmpImagePicker.utils.createPermissionsManager
import kmpImagePicker.utils.rememberGalleryManager
import kotlinx.coroutines.launch

@Composable
fun QrScanner(
    modifier: Modifier,
    flashlightOn: Boolean,
    launchGallery: Boolean,
    onCompletion: (String) -> Unit,
    onGalleryCallBackHandler: (Boolean) -> Unit,
    onFailure: (String) -> Unit
) {
    val coroutineScope = rememberCoroutineScope()
    var permissionRationalDialog by remember { mutableStateOf(value = false) }
    var launchSetting by remember { mutableStateOf(value = false) }

    val permissionsManager = createPermissionsManager(object : PermissionCallback {
        override fun onPermissionStatus(
            permissionType: PermissionType,
            status: PermissionStatus
        ) {
            when (status) {
                PermissionStatus.GRANTED -> {
                    when (permissionType) {
                        PermissionType.GALLERY -> onGalleryCallBackHandler(true)
                    }
                }

                else -> {
                    permissionRationalDialog = true
                }
            }
        }
    })

    val galleryManager = rememberGalleryManager {
        coroutineScope.launch {
            scanImage(it, onCompletion, onFailure)
        }
    }

    if (launchGallery) {
        if (permissionsManager.isPermissionGranted(PermissionType.GALLERY)) {
            galleryManager.launch()
        } else {
            permissionsManager.askPermission(PermissionType.GALLERY)
        }
        onGalleryCallBackHandler(false)
    }

    if (launchSetting) {
        permissionsManager.launchSettings()
        launchSetting = false
    }

    if (permissionRationalDialog) {
        AlertMessageDialog(title = "Permission Required",
            message = "To scan QR code from picture, please grant this permission. You can manage permission in your device settings.",
            positiveButtonText = "Settings",
            negativeButtonText = "Cancel",
            onPositiveClick = {
                permissionRationalDialog = false
                launchSetting = true

            },
            onNegativeClick = {
                permissionRationalDialog = false
            })
    }

    QrCodeScanner(modifier, flashlightOn, onCompletion)
}