package network.chaintech.ui.datetimepicker

import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.BottomSheetDefaults
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.LocalTextStyle
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Shape
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import kotlinx.datetime.LocalDateTime
import network.chaintech.utils.DateTimePickerView
import network.chaintech.utils.MAX
import network.chaintech.utils.MIN
import network.chaintech.utils.now

/***
 * modifier: Modifies the layout of the datetime picker.
 * showDatePicker: Show and hide datetime picker.
 * title: Title displayed above the datetime picker.
 * doneLabel: Label for the "Done" button.
 * titleStyle: Style for the title text.
 * doneLabelStyle: Style for the "Done" label text.
 * minDateTime: Minimum selectable datetime.
 * maxDateTime: Maximum selectable datetime.
 * yearsRange: Initial years range.
 * timeFormat: Format for displaying time (e.g., 24-hour format).
 * height: height of the datetime picker component.
 * rowCount: Number of rows displayed in the picker and it's depending on height also.
 * dateTextStyle: Text style for the datetime display.
 * dateTextColor: Text color for the datetime display.
 * hideHeader: Hide header of picker.
 * containerColor: The color used for the background of datetime picker.
 * shape: The shape of the datetime picker.
 * dateTimePickerView: For bottomsheet and diloag view.
 * dragHandle - Optional visual marker to swipe the bottom sheet.
 * selectorProperties: Properties defining the interaction with the datetime picker.
 * onDoneClick: Callback triggered when the "Done" button is clicked, passing the selected datetime.
 * onDateChangeListener: Callback triggered when the Date is changed, passing the selected datetime.
 * onDismiss: Callback triggered when the datetime picker is dismissed.
 ***/

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun WheelDateTimePickerView(
    modifier: Modifier = Modifier,
    showDatePicker: Boolean = false,
    title: String = "DATE & TIME PICKER",
    doneLabel: String = "Done",
    timeFormat: TimeFormat = TimeFormat.HOUR_24,
    titleStyle: TextStyle = LocalTextStyle.current,
    doneLabelStyle: TextStyle = LocalTextStyle.current,
    startDate: LocalDateTime = LocalDateTime.now(),
    minDate: LocalDateTime = LocalDateTime.MIN(),
    maxDate: LocalDateTime = LocalDateTime.MAX(),
    yearsRange: IntRange? = IntRange(1922, 2122),
    height: Dp,
    rowCount: Int = 3,
    dateTextStyle: TextStyle = MaterialTheme.typography.titleMedium,
    dateTextColor: Color = LocalContentColor.current,
    hideHeader: Boolean = false,
    containerColor: Color = Color.White,
    shape: Shape = RoundedCornerShape(10.dp),
    dateTimePickerView: DateTimePickerView = DateTimePickerView.BOTTOM_SHEET_VIEW,
    dragHandle: @Composable (() -> Unit)? = { BottomSheetDefaults.DragHandle() },
    selectorProperties: SelectorProperties = WheelPickerDefaults.selectorProperties(),
    onDoneClick: (snappedDate: LocalDateTime) -> Unit = {},
    onDateChangeListener: (snappedDate: LocalDateTime) -> Unit = {},
    onDismiss: () -> Unit = {},
) {
    if (dateTimePickerView == DateTimePickerView.BOTTOM_SHEET_VIEW) {
        WheelDateTimePickerBottomSheet(
            modifier = modifier,
            showDatePicker = showDatePicker,
            title = title,
            timeFormat = timeFormat,
            doneLabel = doneLabel,
            titleStyle = titleStyle,
            doneLabelStyle = doneLabelStyle,
            startDate = startDate,
            minDate = minDate,
            maxDate = maxDate,
            yearsRange = yearsRange,
            height = height,
            rowCount = rowCount,
            dateTextStyle = dateTextStyle,
            dateTextColor = dateTextColor,
            hideHeader = hideHeader,
            containerColor = containerColor,
            shape = shape,
            selectorProperties = selectorProperties,
            dragHandle = dragHandle,
            onDoneClick = onDoneClick,
            onDateChangeListener = onDateChangeListener,
            onDismiss = onDismiss
        )
    } else {
        WheelDateTimePickerDialog(
            modifier = modifier,
            showDatePicker = showDatePicker,
            title = title,
            timeFormat = timeFormat,
            doneLabel = doneLabel,
            titleStyle = titleStyle,
            doneLabelStyle = doneLabelStyle,
            startDate = startDate,
            minDate = minDate,
            maxDate = maxDate,
            yearsRange = yearsRange,
            height = height,
            rowCount = rowCount,
            dateTextStyle = dateTextStyle,
            dateTextColor = dateTextColor,
            hideHeader = hideHeader,
            containerColor = containerColor,
            shape = shape,
            selectorProperties = selectorProperties,
            onDoneClick = onDoneClick,
            onDateChangeListener = onDateChangeListener,
            onDismiss = onDismiss
        )
    }
}

