package network.chaintech.ui.datetimepicker

import androidx.compose.runtime.Composable
import androidx.compose.runtime.Immutable
import androidx.compose.runtime.State
import androidx.compose.runtime.rememberUpdatedState
import androidx.compose.ui.graphics.Color
import kotlinx.datetime.LocalDate
import kotlinx.datetime.LocalDateTime
import kotlinx.datetime.LocalTime
import network.chaintech.utils.isLeapYear

object WheelPickerDefaults {
    @Composable
    fun selectorProperties(
        enabled: Boolean = true,
        borderColor: Color = Color(0xFF007AFF).copy(0.7f),
    ): SelectorProperties = DefaultSelectorProperties(
        enabled = enabled,
        borderColor = borderColor,
    )
}

interface SelectorProperties {
    @Composable
    fun enabled(): State<Boolean>

    @Composable
    fun borderColor(): State<Color>

}

@Immutable
internal class DefaultSelectorProperties(
    private val enabled: Boolean,
    private val borderColor: Color,
) : SelectorProperties {

    @Composable
    override fun enabled(): State<Boolean> {
        return rememberUpdatedState(enabled)
    }

    @Composable
    override fun borderColor(): State<Color> {
        return rememberUpdatedState(borderColor)
    }

}

enum class TimeFormat {
    HOUR_24, AM_PM
}

data class Hour(
    val text: String,
    val value: Int,
    val index: Int
)

data class AmPmHour(
    val text: String,
    val value: Int,
    val index: Int
)

fun localTimeToAmPmHour(localTime: LocalTime): Int {
    if (
        isBetween(
            localTime,
            LocalTime(0, 0),
            LocalTime(0, 59)
        )
    ) {
        return localTime.hour + 12
    }

    if (
        isBetween(
            localTime,
            LocalTime(1, 0),
            LocalTime(11, 59)
        )
    ) {
        return localTime.hour
    }

    if (
        isBetween(
            localTime,
            LocalTime(12, 0),
            LocalTime(12, 59)
        )
    ) {
        return localTime.hour
    }

    if (
        isBetween(
            localTime,
            LocalTime(13, 0),
            LocalTime(23, 59)
        )
    ) {
        return localTime.hour - 12
    }

    return localTime.hour
}

fun isBetween(localTime: LocalTime, startTime: LocalTime, endTime: LocalTime): Boolean {
    return localTime in startTime..endTime
}

fun amPmHourToHour24(amPmHour: Int, amPmMinute: Int, amPmValue: AmPmValue): Int {

    return when (amPmValue) {
        AmPmValue.AM -> {
            if (amPmHour == 12 && amPmMinute <= 59) {
                0
            } else {
                amPmHour
            }
        }

        AmPmValue.PM -> {
            if (amPmHour == 12 && amPmMinute <= 59) {
                amPmHour
            } else {
                amPmHour + 12
            }
        }
    }
}

data class Minute(
    val text: String,
    val value: Int,
    val index: Int
)

data class AmPm(
    val text: String,
    val value: AmPmValue,
    val index: Int?
)

enum class AmPmValue {
    AM, PM
}

fun amPmValueFromTime(time: LocalTime): AmPmValue {
    return if (time.hour > 11) AmPmValue.PM else AmPmValue.AM
}

internal sealed class SnappedDate(val snappedLocalDate: LocalDate) {
    data class DayOfMonth(val localDate: LocalDate, val index: Int) :
        SnappedDate(localDate)

    data class Month(val localDate: LocalDate, val index: Int) : SnappedDate(localDate)
    data class Year(val localDate: LocalDate, val index: Int) : SnappedDate(localDate)
}

internal data class DayOfMonth(
    val text: String,
    val value: Int,
    val index: Int
)

data class Month(
    val text: String,
    val value: Int,
    val index: Int
)

data class Year(
    val text: String,
    val value: Int,
    val index: Int
)

private fun calculateDayOfMonths(month: Int, year: Int): List<DayOfMonth> {
    val isLeapYear = isLeapYear(year)

    val month31day = (1..31).map {
        DayOfMonth(
            text = it.toString(),
            value = it,
            index = it - 1
        )
    }
    val month30day = (1..30).map {
        DayOfMonth(
            text = it.toString(),
            value = it,
            index = it - 1
        )
    }
    val month29day = (1..29).map {
        DayOfMonth(
            text = it.toString(),
            value = it,
            index = it - 1
        )
    }
    val month28day = (1..28).map {
        DayOfMonth(
            text = it.toString(),
            value = it,
            index = it - 1
        )
    }

    return when (month) {
        1 -> {
            month31day
        }

        2 -> {
            if (isLeapYear) month29day else month28day
        }

        3 -> {
            month31day
        }

        4 -> {
            month30day
        }

        5 -> {
            month31day
        }

        6 -> {
            month30day
        }

        7 -> {
            month31day
        }

        8 -> {
            month31day
        }

        9 -> {
            month30day
        }

        10 -> {
            month31day
        }

        11 -> {
            month30day
        }

        12 -> {
            month31day
        }

        else -> {
            emptyList()
        }
    }
}

internal sealed class SnappedDateTime(
    val snappedLocalDateTime: LocalDateTime,
    val snappedIndex: Int
) {
    data class DayOfMonth(val localDateTime: LocalDateTime, val index: Int) :
        SnappedDateTime(localDateTime, index)

    data class Month(val localDateTime: LocalDateTime, val index: Int) :
        SnappedDateTime(localDateTime, index)

    data class Year(val localDateTime: LocalDateTime, val index: Int) :
        SnappedDateTime(localDateTime, index)

    data class Hour(val localDateTime: LocalDateTime, val index: Int) :
        SnappedDateTime(localDateTime, index)

    data class Minute(val localDateTime: LocalDateTime, val index: Int) :
        SnappedDateTime(localDateTime, index)
}

internal sealed class SnappedTime(val snappedLocalTime: LocalTime) {
    data class Hour(val localTime: LocalTime, val index: Int) : SnappedTime(localTime)
    data class Minute(val localTime: LocalTime, val index: Int) : SnappedTime(localTime)
}