package network.chaintech.ui.timepicker

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.LocalTextStyle
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import kotlinx.datetime.LocalTime
import network.chaintech.utils.MAX
import network.chaintech.utils.MIN
import network.chaintech.utils.SelectorProperties
import network.chaintech.utils.TimeFormat
import network.chaintech.utils.WheelPickerDefaults
import network.chaintech.utils.now

object WheelTimePickerComponent {

    /***
     * modifier: Modifies the layout of the time picker.
     * title: Title displayed above the time picker.
     * doneLabel: Label for the "Done" button.
     * titleStyle: Style for the title text.
     * doneLabelStyle: Style for the "Done" label text.
     * startTime: Initial time selected in the picker.
     * minTime: Minimum selectable time.
     * maxTime: Maximum selectable time.
     * timeFormat: Format for displaying time (e.g., 24-hour format).
     * height: height of the time picker component.
     * rowCount: Number of rows displayed in the picker and it's depending on height also.
     * textStyle: Text style for the time display.
     * textColor: Text color for the time display.
     * hideHeader: Hide header of picker.
     * selectorProperties: Properties defining the interaction with the time picker.
     * onDoneClick: Callback triggered when the "Done" button is clicked, passing the selected time.
     * onTimeChangeListener: Callback triggered when the time is changed, passing the selected time.
     ***/

    @Composable
    fun WheelTimePicker(
        modifier: Modifier = Modifier,
        title: String = "TIME PICKER",
        doneLabel: String = "Done",
        titleStyle: TextStyle = LocalTextStyle.current,
        doneLabelStyle: TextStyle = LocalTextStyle.current,
        startTime: LocalTime = LocalTime.now(),
        minTime: LocalTime = LocalTime.MIN(),
        maxTime: LocalTime = LocalTime.MAX(),
        timeFormat: TimeFormat = TimeFormat.HOUR_24,
        height: Dp,
        rowCount: Int = 3,
        textStyle: TextStyle = MaterialTheme.typography.titleMedium,
        textColor: Color = LocalContentColor.current,
        hideHeader: Boolean = false,
        selectorProperties: SelectorProperties = WheelPickerDefaults.selectorProperties(),
        onDoneClick: (snappedDate: LocalTime) -> Unit = {},
        onTimeChangeListener: (snappedDate: LocalTime) -> Unit = {},
    ) {
        var selectedDate by remember { mutableStateOf(LocalTime.now()) }

        LaunchedEffect(selectedDate) {
            if (hideHeader) {
                onTimeChangeListener(selectedDate)
            }
        }

        Column(modifier = modifier) {
            if (!hideHeader) {
                Row(
                    modifier = Modifier.fillMaxWidth().padding(horizontal = 20.dp),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    Text(
                        text = title,
                        style = titleStyle,
                        modifier = Modifier.weight(1f)
                    )
                    Text(
                        text = doneLabel,
                        style = doneLabelStyle,
                        modifier = Modifier
                            .noRippleEffect {
                                onDoneClick(selectedDate)
                            }
                    )
                }
                HorizontalDivider(
                    modifier = Modifier.padding(top = 12.dp),
                    thickness = (0.5).dp,
                    color = Color.LightGray
                )
            }
            DefaultWheelTimePicker(
                textColor = textColor,
                timeFormat = timeFormat,
                selectorProperties = selectorProperties,
                rowCount = rowCount,
                height = height,
                modifier = Modifier.padding(top = 14.dp, bottom = 14.dp),
                startTime = startTime,
                minTime = minTime,
                maxTime = maxTime,
                textStyle = textStyle,
                onSnappedTime = { snappedTime, _ ->
                    selectedDate = snappedTime.snappedLocalTime
                    snappedTime.snappedIndex
                }
            )
        }
    }

}
