package network.chaintech.ui.datetimepicker

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import kotlinx.datetime.DateTimeUnit
import kotlinx.datetime.LocalDateTime
import network.chaintech.utils.MAX
import network.chaintech.utils.MIN
import network.chaintech.utils.now
import network.chaintech.utils.truncateTo
import network.chaintech.utils.withDayOfMonth
import network.chaintech.utils.withHour
import network.chaintech.utils.withMinute
import network.chaintech.utils.withMonth
import network.chaintech.utils.withYear

@Composable
internal fun DefaultWheelDateTimePicker(
    modifier: Modifier = Modifier,
    startDateTime: LocalDateTime = LocalDateTime.now(),
    minDateTime: LocalDateTime = LocalDateTime.MIN(),
    maxDateTime: LocalDateTime = LocalDateTime.MAX(),
    yearsRange: IntRange? = IntRange(1922, 2122),
    timeFormat: TimeFormat = TimeFormat.HOUR_24,
    height: Dp = 128.dp,
    rowCount: Int = 3,
    textStyle: TextStyle = MaterialTheme.typography.titleMedium,
    textColor: Color = LocalContentColor.current,
    selectorProperties: SelectorProperties = WheelPickerDefaults.selectorProperties(),
    onSnappedDateTime: (snappedDateTime: SnappedDateTime) -> Int? = { _ -> null }
) {
    var snappedDateTime by remember { mutableStateOf(startDateTime.truncateTo(DateTimeUnit.MINUTE)) }

    val yearTexts = yearsRange?.map { it.toString() } ?: listOf()

    Box(modifier = modifier, contentAlignment = Alignment.Center) {
        if (selectorProperties.enabled().value) {
            HorizontalDivider(
                modifier = Modifier.padding(bottom = (height / rowCount)),
                thickness = (0.5).dp,
                color = selectorProperties.borderColor().value
            )
            HorizontalDivider(
                modifier = Modifier.padding(top = (height / rowCount)),
                thickness = (0.5).dp,
                color = selectorProperties.borderColor().value
            )
        }
        Row(modifier = Modifier.fillMaxWidth()) {
            DefaultWheelDatePicker(
                modifier = Modifier.weight(1f).padding(start = 12.dp),
                startDate = startDateTime.date,
                yearsRange = yearsRange,
                height = height,
                rowCount = rowCount,
                textStyle = textStyle,
                textColor = textColor
            ) { snappedDate ->

                val newDateTime = when (snappedDate) {
                    is SnappedDate.DayOfMonth -> {
                        snappedDateTime.withDayOfMonth(snappedDate.snappedLocalDate.dayOfMonth)
                    }

                    is SnappedDate.Month -> {
                        snappedDateTime.withMonth(snappedDate.snappedLocalDate.monthNumber)
                    }

                    is SnappedDate.Year -> {
                        snappedDateTime.withYear(snappedDate.snappedLocalDate.year)
                    }

                }
                if (newDateTime.compareTo(minDateTime) >= 0 && newDateTime.compareTo(
                        maxDateTime
                    ) <= 0
                ) {
                    snappedDateTime = newDateTime
                }

                return@DefaultWheelDatePicker when (snappedDate) {
                    is SnappedDate.DayOfMonth -> {
                        onSnappedDateTime(
                            SnappedDateTime.DayOfMonth(
                                snappedDateTime,
                                snappedDateTime.dayOfMonth - 1
                            )
                        )
                        snappedDateTime.dayOfMonth - 1
                    }

                    is SnappedDate.Month -> {
                        onSnappedDateTime(
                            SnappedDateTime.Month(
                                snappedDateTime,
                                snappedDateTime.monthNumber - 1
                            )
                        )
                        snappedDateTime.monthNumber - 1
                    }

                    is SnappedDate.Year -> {
                        onSnappedDateTime(
                            SnappedDateTime.Year(
                                snappedDateTime,
                                yearTexts.indexOf(snappedDateTime.year.toString())
                            )
                        )
                        yearTexts.indexOf(snappedDateTime.year.toString())
                    }

                }
            }
            Spacer(modifier = (Modifier.weight(0.1f)))
            DefaultWheelTimePicker(
                startTime = startDateTime.time,
                timeFormat = timeFormat,
                height = height,
                rowCount = rowCount,
                textStyle = textStyle,
                textColor = textColor,
                selectorProperties = WheelPickerDefaults.selectorProperties(
                    enabled = false
                ),
                onSnappedTime = { snappedTime, timeFormat ->

                    val newDateTime = when (snappedTime) {
                        is SnappedTime.Hour -> {
                            snappedDateTime.withHour(snappedTime.snappedLocalTime.hour)
                        }

                        is SnappedTime.Minute -> {
                            snappedDateTime.withMinute(snappedTime.snappedLocalTime.minute)
                        }
                    }

                    if (newDateTime.compareTo(minDateTime) >= 0 && newDateTime.compareTo(
                            maxDateTime
                        ) <= 0
                    ) {
                        snappedDateTime = newDateTime
                    }

                    return@DefaultWheelTimePicker when (snappedTime) {
                        is SnappedTime.Hour -> {
                            onSnappedDateTime(
                                SnappedDateTime.Hour(
                                    snappedDateTime,
                                    snappedDateTime.hour
                                )
                            )
                            if (timeFormat == TimeFormat.HOUR_24) snappedDateTime.hour else
                                localTimeToAmPmHour(snappedDateTime.time) - 1
                        }

                        is SnappedTime.Minute -> {
                            onSnappedDateTime(
                                SnappedDateTime.Minute(
                                    snappedDateTime,
                                    snappedDateTime.minute
                                )
                            )
                            snappedDateTime.minute
                        }
                    }
                },
                modifier = Modifier.weight(1f).padding(end = 12.dp)
            )
        }
    }
}