package network.chaintech.composeMultiplatformScreenshot

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.window.ComposeUIViewController
import kotlinx.cinterop.ExperimentalForeignApi
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import network.chaintech.composeMultiplatformScreenshot.extension.toImageBitmap
import network.chaintech.composeMultiplatformScreenshot.model.SizeModel
import network.chaintech.composeMultiplatformScreenshot.utility.shareScreenshot
import platform.CoreGraphics.CGRectMake
import platform.UIKit.UIGraphicsImageRenderer
import platform.UIKit.UIImage
import platform.UIKit.UIView

var screenShotSize: SizeModel? = null

@Composable
fun TakeScreenShot(
    controller: ScreenCaptureController,
    shareImage: Boolean,
    onCaptured: (ImageBitmap?, Throwable?) -> Unit,
    content: @Composable () -> Unit
) {
    if (controller.isCaptureScreen.value) {
        screenshotComposable(shareImage, onCaptured) {
            Box(modifier = Modifier.verticalScroll(rememberScrollState())) {
                content()
            }
        }
        controller.captured()
    }
}

private fun screenshotComposable(
    shareImage: Boolean,
    onCaptured: (ImageBitmap?, Throwable?) -> Unit,
    content: @Composable () -> Unit
) {
    CoroutineScope(Dispatchers.Main).launch {
        screenShotSize?.let {
            val contentView = getUIView(content = content, viewSize = it)
            val screenShotImage = imageFromUIView(contentView)
            onCaptured(screenShotImage.toImageBitmap(), null)
            if (shareImage) {
                shareScreenshot(screenShotImage)
            }
        }
    }
}


@OptIn(ExperimentalForeignApi::class)
private fun imageFromUIView(contentView: UIView): UIImage {
    val renderer = UIGraphicsImageRenderer(bounds = contentView.bounds)
    val image = renderer.imageWithActions {
        contentView.drawViewHierarchyInRect(contentView.bounds, afterScreenUpdates = true)
    }
    return image
}
@OptIn(ExperimentalForeignApi::class)
private fun getUIView(content: @Composable () -> Unit, viewSize: SizeModel): UIView {
    val viewController = ComposeUIViewController {
        content()
    }
    viewController.view.layer.bounds =
        CGRectMake(x = 0.0, y = 0.0, width = viewSize.width, height = viewSize.height)
    viewController.view.setNeedsLayout()
    viewController.view.layoutIfNeeded()
    return viewController.view
}
