package chaintech.videoplayer.ui

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.animation.core.tween
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.foundation.Image
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.size
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.FastForward
import androidx.compose.material.icons.filled.FastRewind
import androidx.compose.material.icons.filled.Pause
import androidx.compose.material.icons.filled.PlayArrow
import androidx.compose.material3.Icon
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.scale
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.unit.Dp
import chaintech.videoplayer.model.PlayerConfig
import kotlinx.coroutines.delay
import org.jetbrains.compose.resources.DrawableResource
import org.jetbrains.compose.resources.ExperimentalResourceApi
import org.jetbrains.compose.resources.painterResource

@OptIn(ExperimentalResourceApi::class)
@Composable
fun PlayView(
    playerConfig: PlayerConfig,
    isPause: Boolean,
    onPauseToggle: (() -> Unit),
    onBackwardToggle: (() -> Unit),
    onForwardToggle: (() -> Unit),
    showControls: Boolean
) {
        AnimatedVisibility(
            modifier = Modifier,
            visible = showControls,
            enter = fadeIn(),
            exit = fadeOut()
        ) {
            Box(
                modifier = Modifier
                    .fillMaxSize(),
                contentAlignment = Alignment.Center
            ) {
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.SpaceAround
                ){
                    if (playerConfig.enableFastForwardBackward) {
                        if (playerConfig.fastBackwardIcon != null) {
                            AnimatedClickableIcon(
                                painterRes = playerConfig.fastBackwardIcon,
                                contentDescription = "Fast Backward",
                                tint = Color.White,
                                iconSize = playerConfig.pauseResumeIconSize,
                                onClick = {onBackwardToggle()}
                            )
                        }else {
                            AnimatedClickableIcon(
                                imageVector = Icons.Filled.FastRewind,
                                contentDescription = "Fast Backward",
                                tint = Color.White,
                                iconSize = playerConfig.pauseResumeIconSize,
                                onClick = {onBackwardToggle()}
                            )
                        }
                    }

                    if ((playerConfig.playIcon != null) && (playerConfig.pauseIcon != null)) {
                        AnimatedClickableIcon(
                            painterRes = if (isPause) { playerConfig.playIcon } else { playerConfig.pauseIcon },
                            contentDescription = "Play/Pause",
                            tint = Color.White,
                            iconSize = playerConfig.pauseResumeIconSize,
                            onClick = {onPauseToggle()}
                        )
                    } else {
                        AnimatedClickableIcon(
                            imageVector = if (isPause) Icons.Filled.PlayArrow else Icons.Filled.Pause,
                            contentDescription = "Play/Pause",
                            tint = Color.White,
                            iconSize = playerConfig.pauseResumeIconSize,
                            onClick = {onPauseToggle()}
                        )
                    }

                    if (playerConfig.enableFastForwardBackward) {
                        if (playerConfig.fastForwardIcon != null) {
                            AnimatedClickableIcon(
                                painterRes = playerConfig.fastForwardIcon,
                                contentDescription = "Fast Forward",
                                tint = Color.White,
                                iconSize = playerConfig.pauseResumeIconSize,
                                onClick = {onForwardToggle()}
                            )
                        }else {
                            AnimatedClickableIcon(
                                imageVector = Icons.Filled.FastForward,
                                contentDescription = "Fast Forward",
                                tint = Color.White,
                                iconSize = playerConfig.pauseResumeIconSize,
                                onClick = { onForwardToggle() }
                            )
                        }
                    }
                }
            }
        }
}

@OptIn(ExperimentalResourceApi::class)
@Composable
private fun AnimatedClickableIcon(
    painterRes: DrawableResource? = null,
    imageVector: ImageVector? = null,
    contentDescription: String?,
    tint: Color,
    iconSize: Dp,
    animationDuration: Int = 300,
    onClick: () -> Unit
) {
    var isClicked by remember { mutableStateOf(false) }

    val scale by animateFloatAsState(
        targetValue = if (isClicked) 0.8f else 1f,
        animationSpec = tween(durationMillis = animationDuration)
    )
    LaunchedEffect(isClicked) {
        if (isClicked) {
            delay(animationDuration.toLong())
            isClicked = false
        }
    }

    Box(
        modifier = Modifier
            .size(iconSize)
            .scale(scale)
            .pointerInput(Unit) {
                detectTapGestures { _ ->
                    isClicked = true
                    onClick()
                }
            }
    ){
        imageVector?.let {
            Icon(
                imageVector = imageVector,
                contentDescription = contentDescription,
                tint = tint,
                modifier = Modifier
                    .fillMaxSize()
            )
        }

        painterRes?.let {
            Image(
                painter = painterResource(painterRes),
                contentDescription = contentDescription,
                contentScale = ContentScale.Fit,
                modifier = Modifier
                    .fillMaxSize()
            )
        }
    }

}