package chaintech.videoplayer.ui

import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.VerticalPager
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.runtime.snapshotFlow
import androidx.compose.ui.Modifier
import chaintech.videoplayer.model.PlayerConfig
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.distinctUntilChanged
import org.jetbrains.compose.resources.ExperimentalResourceApi

@OptIn(ExperimentalFoundationApi::class, ExperimentalResourceApi::class)
@Composable
fun ReelsPlayerView(modifier : Modifier = Modifier,
                    urls:  List<String>,
                    playerConfig: PlayerConfig = PlayerConfig()
) {
    val pagerState = rememberPagerState(pageCount = {
        urls.size
    })

    LaunchedEffect(key1 = pagerState) {
        snapshotFlow { pagerState.currentPage }.distinctUntilChanged().collect { page ->
            pagerState.animateScrollToPage(page)
        }
    }
    var showControls by remember { mutableStateOf(true) }

    if(playerConfig.autoHideControl) {
        LaunchedEffect(showControls) {
            if (showControls) {
                delay(timeMillis = (playerConfig.controlHideInterval * 1000).toLong())
                showControls = false
            }
        }
    }

    if (playerConfig.reelVerticalScrolling) {
        VerticalPager(
            state = pagerState,
        ) { page ->
            var isPause by remember { mutableStateOf(false) }
            ReelVideoPlayer(
                modifier = modifier,
                url = urls[page],
                playerConfig = playerConfig,
                isPause = if (pagerState.currentPage == page) { isPause } else { true },
                onPauseToggle = { isPause = isPause.not() },
                showControls = showControls,
                onShowControlsToggle = { showControls = showControls.not() }
            )
        }
    } else {
        HorizontalPager(
            state = pagerState
        ) { page ->
            var isPause by remember { mutableStateOf(false) }
            ReelVideoPlayer(
                modifier = modifier,
                url = urls[page],
                playerConfig = playerConfig,
                isPause = if (pagerState.currentPage == page) { isPause } else { true },
                onPauseToggle = { isPause = isPause.not() },
                showControls = showControls,
                onShowControlsToggle = { showControls = showControls.not() }
            )
        }
    }
}

@OptIn(ExperimentalResourceApi::class)
@Composable
private fun ReelVideoPlayer(modifier : Modifier = Modifier,
                            url: String,
                            playerConfig: PlayerConfig = PlayerConfig(),
                            isPause: Boolean,
                            onPauseToggle: (() -> Unit),
                            showControls: Boolean,
                            onShowControlsToggle: (() -> Unit)) {
    if (playerConfig.showSeekBar) {
        VideoPlayerWithControl(
            modifier = modifier,
            url = url,
            playerConfig = playerConfig,
            isPause = isPause,
            onPauseToggle = onPauseToggle,
            showControls = showControls,
            onShowControlsToggle = onShowControlsToggle
        )
    } else {
        VideoPlayerWithoutControl(
            modifier = modifier,
            url = url,
            playerConfig = playerConfig,
            isPause = isPause,
            onPauseToggle = onPauseToggle,
            showControls = showControls,
            onShowControlsToggle = onShowControlsToggle
        )
    }
}
