package chaintech.videoplayer.ui

import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.VerticalPager
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.runtime.snapshotFlow
import androidx.compose.ui.Modifier
import chaintech.videoplayer.model.SeekBarConfig
import kotlinx.coroutines.flow.distinctUntilChanged
import org.jetbrains.compose.resources.DrawableResource
import org.jetbrains.compose.resources.ExperimentalResourceApi

@OptIn(ExperimentalFoundationApi::class, ExperimentalResourceApi::class)
@Composable
fun ReelsPlayerView(modifier : Modifier = Modifier,
                    urls:  List<String>,
                    enablePauseResume: Boolean = true,
                    showSeekBar: Boolean = false,
                    verticalScrolling: Boolean = true,
                    playIcon: DrawableResource? = null,
                    pauseIcon: DrawableResource? = null,
                    seekBarConfig: SeekBarConfig = SeekBarConfig()) {
    val pagerState = rememberPagerState(pageCount = {
        urls.size
    })

    LaunchedEffect(key1 = pagerState) {
        snapshotFlow { pagerState.currentPage }.distinctUntilChanged().collect { page ->
            pagerState.animateScrollToPage(page)
        }
    }
    var showControls by remember { mutableStateOf(true) }

    if (verticalScrolling) {
        VerticalPager(
            state = pagerState,
            beyondBoundsPageCount = 1
        ) { page ->
            var isPause by remember { mutableStateOf(false) }

            ReelVideoPlayer(
                modifier = modifier,
                url = urls[page],
                enablePauseResume = enablePauseResume,
                showSeekBar = showSeekBar,
                isPause = if (pagerState.currentPage == page) { isPause } else { true },
                onPauseToggle = { isPause = isPause.not() },
                showControls = showControls,
                onShowControlsToggle = { showControls = showControls.not() },
                playIcon = playIcon,
                pauseIcon = pauseIcon,
                seekBarConfig = seekBarConfig
            )
        }
    } else {
        HorizontalPager(
            state = pagerState,
            beyondBoundsPageCount = 0
        ) { page ->
            var isPause by remember { mutableStateOf(false) }

            ReelVideoPlayer(
                modifier = modifier,
                url = urls[page],
                enablePauseResume = enablePauseResume,
                showSeekBar = showSeekBar,
                isPause = if (pagerState.currentPage == page) { isPause } else { true },
                onPauseToggle = { isPause = isPause.not() },
                showControls = showControls,
                onShowControlsToggle = { showControls = showControls.not() },
                playIcon = playIcon,
                pauseIcon = pauseIcon,
                seekBarConfig = seekBarConfig
            )
        }
    }
}

@OptIn(ExperimentalResourceApi::class)
@Composable
private fun ReelVideoPlayer(modifier : Modifier = Modifier,
                            url: String,
                            enablePauseResume: Boolean,
                            showSeekBar: Boolean = false,
                            isPause: Boolean,
                            onPauseToggle: (() -> Unit),
                            showControls: Boolean,
                            onShowControlsToggle: (() -> Unit),
                            playIcon: DrawableResource? = null,
                            pauseIcon: DrawableResource? = null,
                            seekBarConfig: SeekBarConfig) {
    if (showSeekBar) {
        VideoPlayerWithControl(
            modifier = modifier,
            url = url,
            enablePauseResume = enablePauseResume,
            isPause = isPause,
            onPauseToggle = onPauseToggle,
            showControls = showControls,
            onShowControlsToggle = onShowControlsToggle,
            playIcon = playIcon,
            pauseIcon = pauseIcon,
            seekBarConfig = seekBarConfig
        )
    } else {
        VideoPlayerWithoutControl(
            modifier = modifier,
            url = url,
            enablePauseResume = enablePauseResume,
            isPause = isPause,
            onPauseToggle = onPauseToggle,
            playIcon = playIcon,
            pauseIcon = pauseIcon
        )
    }
}
