package chaintech.videoplayer.ui

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Slider
import androidx.compose.material3.SliderDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.unit.dp
import chaintech.videoplayer.extension.formatMinSec
import chaintech.videoplayer.model.PlayerConfig
import chaintech.videoplayer.util.CMPPlayerWithControls

@Composable
fun VideoPlayerWithControl(modifier : Modifier,
                           url: String,
                           playerConfig: PlayerConfig,
                           isPause: Boolean,
                           onPauseToggle: (() -> Unit),
                           showControls: Boolean,
                           onShowControlsToggle: (() -> Unit)
) {
    var totalT by remember { mutableStateOf(0) }
    var currentT by remember { mutableStateOf(0) }

    var isSliding by remember { mutableStateOf(false) }
    var sliderTime: Int? by remember { mutableStateOf(null) }

    Box(
        modifier = modifier
            .pointerInput(Unit) {
                detectTapGestures { _ ->
                    onShowControlsToggle()
                }
            }
    ) {

        CMPPlayerWithControls(
            modifier = modifier,
            url = url,
            isPause = isPause,
            totalTime = { totalT = it },
            currentTime = {
                if (!isSliding) {
                    currentT = it
                    sliderTime = null
                }
            },
            isSliding = isSliding,
            sliderTime = sliderTime
        )

        Column(
            modifier = Modifier.align(Alignment.BottomStart)
                .padding(bottom = playerConfig.seekBarBottomPadding)
        ) {
            AnimatedVisibility(
                modifier = Modifier,
                visible = showControls,
                enter = fadeIn(),
                exit = fadeOut()
            ) {
                Row(
                    modifier = Modifier.fillMaxWidth()
                        .padding(horizontal = 16.dp),
                    verticalAlignment = Alignment.Top
                ) {
                    Column(
                        modifier = Modifier.weight(1f)
                    ) {
                        Slider(
                            modifier = Modifier.fillMaxWidth().height(25.dp),
                            value = currentT.toFloat(),
                            onValueChange = {
                                sliderTime = null
                                isSliding = true
                                currentT = it.toInt()
                            },
                            valueRange = 0f..totalT.toFloat(),
                            onValueChangeFinished = {
                                isSliding = false
                                sliderTime = currentT
                            },
                            colors = SliderDefaults.colors(
                                thumbColor = playerConfig.thumbColor,
                                inactiveTrackColor = playerConfig.inactiveTrackColor,
                                activeTrackColor = playerConfig.activeTrackColor
                            )
                        )

                        if (playerConfig.showDuration) {
                            Row(
                                modifier = Modifier
                                    .fillMaxWidth(),
                                horizontalArrangement = Arrangement.SpaceBetween
                            ) {
                                Text(
                                    modifier = Modifier,
                                    text = currentT.formatMinSec(),
                                    color = playerConfig.textColor
                                )
                                Spacer(Modifier.weight(1f))
                                Text(
                                    modifier = Modifier,
                                    text = totalT.formatMinSec(),
                                    color = playerConfig.textColor
                                )
                            }
                        }
                    }
                }
            }
        }
        if (playerConfig.enablePauseResume) {
            PlayView(
                playerConfig = playerConfig,
                isPause = isPause,
                onPauseToggle = onPauseToggle,
                showControls = showControls
            )
        }
    }
}