package chaintech.videoplayer.ui

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.foundation.Image
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Pause
import androidx.compose.material.icons.filled.PlayArrow
import androidx.compose.material3.Icon
import androidx.compose.material3.Slider
import androidx.compose.material3.SliderDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.unit.dp
import chaintech.videoplayer.extension.formatMinSec
import chaintech.videoplayer.model.SeekBarConfig
import chaintech.videoplayer.util.CMPPlayerWithControls
import org.jetbrains.compose.resources.DrawableResource
import org.jetbrains.compose.resources.ExperimentalResourceApi
import org.jetbrains.compose.resources.painterResource

@OptIn(ExperimentalResourceApi::class)
@Composable
fun VideoPlayerWithControl(modifier : Modifier,
                           url: String,
                           enablePauseResume: Boolean,
                           isPause: Boolean,
                           onPauseToggle: (() -> Unit),
                           showControls: Boolean,
                           onShowControlsToggle: (() -> Unit),
                           playIcon: DrawableResource? = null,
                           pauseIcon: DrawableResource? = null,
                           seekBarConfig: SeekBarConfig
) {
    var totalT by remember { mutableStateOf(0) }
    var currentT by remember { mutableStateOf(0) }

    var isSliding by remember { mutableStateOf(false) }
    var sliderTime: Int? by remember { mutableStateOf(null) }

    Box(
        modifier = modifier
            .pointerInput(Unit) {
                detectTapGestures { _ ->
                    onShowControlsToggle()
                }
            }
    ) {

        CMPPlayerWithControls(
            modifier = modifier,
            url = url,
            isPause = isPause,
            totalTime = { totalT = it },
            currentTime = {
                if (!isSliding) {
                    currentT = it
                    sliderTime = null
                }
            },
            isSliding = isSliding,
            sliderTime = sliderTime
        )

        Column(
            modifier = Modifier.align(Alignment.BottomStart)
                .padding(bottom = seekBarConfig.seekBarBottomPadding)
        ) {
            AnimatedVisibility(
                modifier = Modifier,
                visible = showControls,
                enter = fadeIn(),
                exit = fadeOut()
            ) {
                Row(
                    modifier = Modifier.fillMaxWidth()
                        .padding(horizontal = 16.dp),
                    verticalAlignment = Alignment.Top
                ) {

                    if (enablePauseResume) {
                        Box(
                            modifier = Modifier
                                .height(40.dp)
                                .width(30.dp)
                                .pointerInput(Unit) {
                                    detectTapGestures { _ ->
                                        onPauseToggle()
                                    }
                                }
                                .padding(top = 10.dp)
                        ) {
                            if ((playIcon != null) && (pauseIcon != null)) {
                                Image(
                                    painter = if (isPause) painterResource(playIcon) else painterResource(
                                        pauseIcon
                                    ),
                                    contentDescription = "Play/Pause",
                                    contentScale = ContentScale.Fit,
                                    modifier = Modifier
                                        .fillMaxSize()
                                )
                            } else {
                                Icon(
                                    imageVector = if (isPause) Icons.Filled.PlayArrow else Icons.Filled.Pause,
                                    contentDescription = "Play/Pause",
                                    tint = Color.White,
                                    modifier = Modifier
                                        .fillMaxSize()
                                )
                            }
                        }

                        Spacer(modifier = Modifier.width(10.dp))
                    }

                    Column(
                        modifier = Modifier.weight(1f)
                    ) {
                        Box(
                            modifier = Modifier.fillMaxWidth()
                        ) {

                            Slider(
                                modifier = Modifier.fillMaxWidth(),
                                value = currentT.toFloat(),
                                onValueChange = {
                                    sliderTime = null
                                    isSliding = true
                                    currentT = it.toInt()
                                },
                                valueRange = 0f..totalT.toFloat(),
                                onValueChangeFinished = {
                                    isSliding = false
                                    sliderTime = currentT
                                },
                                colors = SliderDefaults.colors(
                                    thumbColor = seekBarConfig.thumbColor,
                                    inactiveTrackColor = seekBarConfig.inactiveTrackColor,
                                    activeTrackColor = seekBarConfig.activeTrackColor
                                )
                            )
                        }
                        if (seekBarConfig.showDuration) {
                            Row(
                                modifier = Modifier
                                    .fillMaxWidth(),
                                horizontalArrangement = Arrangement.SpaceBetween
                            ) {
                                Text(
                                    modifier = Modifier,
                                    text = currentT.formatMinSec(),
                                    color = seekBarConfig.textColor
                                )
                                Spacer(Modifier.weight(1f))
                                Text(
                                    modifier = Modifier,
                                    text = totalT.formatMinSec(),
                                    color = seekBarConfig.textColor
                                )
                            }
                        }
                    }
                }
            }
        }
    }
}