package chaintech.videoplayer.util

import androidx.compose.foundation.layout.Box
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.ui.Modifier
import androidx.compose.ui.viewinterop.AndroidView
import androidx.media3.common.Player
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive

@Composable
actual fun CMPPlayerWithControls(
    modifier: Modifier,
    url: String,
    isPause: Boolean,
    totalTime: ((Int) -> Unit),
    currentTime: ((Int) -> Unit),
    isSliding: Boolean,
    sliderTime: Int?
) {
    val exoPlayer = rememberExoPlayerWithLifecycle(url)
    val playerView = rememberPlayerView(exoPlayer)

    LaunchedEffect(exoPlayer) {
        while (isActive) {
            currentTime(exoPlayer.currentPosition.coerceAtLeast(0L).toInt())
            delay(1000) // Delay for 1 second
        }
    }
    LaunchedEffect(playerView) {
        playerView.keepScreenOn = true
    }
    Box {
        AndroidView(
            factory = { playerView },
            modifier = modifier,
            update = {
                exoPlayer.playWhenReady = !isPause
                sliderTime?.let {
                    exoPlayer.seekTo(it.toLong())
                }
            }
        )

        DisposableEffect(key1 = Unit) {
            val listener = object : Player.Listener {
                override fun onEvents(
                    player: Player, events: Player.Events
                ) {
                    super.onEvents(player, events)
                    if (!isSliding) {

                        totalTime(player.duration.coerceAtLeast(0L).toInt())
                        currentTime(player.currentPosition.coerceAtLeast(0L).toInt())
                    }
                }
            }

            exoPlayer.addListener(listener)

            onDispose {
                exoPlayer.removeListener(listener)
                exoPlayer.release()
            }
        }
    }

    DisposableEffect(url) {
        onDispose {
            exoPlayer.release()
            playerView.keepScreenOn = false
        }
    }
}