package chaintech.network.connectivitymonitor

import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import platform.Network.nw_interface_type_cellular
import platform.Network.nw_interface_type_wifi
import platform.Network.nw_path_get_status
import platform.Network.nw_path_monitor_cancel
import platform.Network.nw_path_monitor_create
import platform.Network.nw_path_monitor_set_queue
import platform.Network.nw_path_monitor_set_update_handler
import platform.Network.nw_path_monitor_start
import platform.Network.nw_path_monitor_t
import platform.Network.nw_path_status_satisfied
import platform.Network.nw_path_uses_interface_type
import platform.darwin.DISPATCH_QUEUE_PRIORITY_BACKGROUND
import platform.darwin.dispatch_get_global_queue
import io.ktor.client.engine.ios.Ios
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json
import io.ktor.client.request.get
import io.ktor.client.HttpClient
import io.ktor.client.plugins.HttpTimeout


object HttpClientProvider {
    val client: HttpClient = HttpClient(Ios) {
        install(ContentNegotiation) {
            json(Json {
                prettyPrint = true
                isLenient = true
                ignoreUnknownKeys = true
            })
        }

        install(HttpTimeout) {
            requestTimeoutMillis = 3000
        }
    }
}

actual class ConnectivityMonitor private constructor() {
    private val monitor: nw_path_monitor_t = nw_path_monitor_create()
    private val queue = dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_BACKGROUND.toLong(), 0.toULong())
    private val _status = MutableStateFlow(ConnectivityStatus.DETERMINING)
    actual val status: StateFlow<ConnectivityStatus> get() = _status
    private var debounceJob: Job? = null
    private var currentConnectionType: iOSConnectionType = iOSConnectionType.NONE
    enum class iOSConnectionType {
        WIFI, CELLULAR, NONE
    }

    init {
        startMonitoring()
    }

    actual fun startMonitoring() {
        nw_path_monitor_set_update_handler(monitor) { path ->
            if (nw_path_get_status(path) == nw_path_status_satisfied) {
                if (nw_path_uses_interface_type(path, nw_interface_type_wifi)) {
                    debounceUpdate {
                        checkInternetConnection { hasInternet ->
                            currentConnectionType = iOSConnectionType.WIFI
                            updateStatus(if (hasInternet) ConnectivityStatus.CONNECTED_VIA_WIFI else ConnectivityStatus.CONNECTED_VIA_WIFI_WITHOUT_INTERNET)
                        }
                    }
                } else if (nw_path_uses_interface_type(path, nw_interface_type_cellular)) {
                    debounceUpdate {
                        checkInternetConnection { hasInternet ->
                            currentConnectionType = iOSConnectionType.CELLULAR
                            updateStatus(if (hasInternet) ConnectivityStatus.CONNECTED_VIA_CELLULAR else ConnectivityStatus.CONNECTED_VIA_CELLULAR_WITHOUT_INTERNET)
                        }
                    }
                } else {
                    debounceUpdate {
                        checkInternetConnection { hasInternet ->
                            currentConnectionType = iOSConnectionType.NONE
                            updateStatus(if (hasInternet) ConnectivityStatus.CONNECTED else ConnectivityStatus.NOT_CONNECTED)
                        }
                    }
                }
            } else {
                debounceUpdate {
                    checkInternetConnection { hasInternet ->
                        currentConnectionType = iOSConnectionType.NONE
                        updateStatus(if (hasInternet) ConnectivityStatus.CONNECTED else ConnectivityStatus.NOT_CONNECTED)
                    }
                }
            }
        }
        nw_path_monitor_set_queue(monitor, queue)
        nw_path_monitor_start(monitor)
    }

    actual fun stopMonitoring() {
        nw_path_monitor_cancel(monitor)
    }

    actual fun refresh() {
        checkInternetConnection { hasInternet ->
            when (currentConnectionType) {
                iOSConnectionType.WIFI -> {
                    updateStatus(if (hasInternet) ConnectivityStatus.CONNECTED_VIA_WIFI else ConnectivityStatus.CONNECTED_VIA_WIFI_WITHOUT_INTERNET)
                }
                iOSConnectionType.CELLULAR -> {
                    updateStatus(if (hasInternet) ConnectivityStatus.CONNECTED_VIA_CELLULAR else ConnectivityStatus.CONNECTED_VIA_CELLULAR_WITHOUT_INTERNET)
                }
                iOSConnectionType.NONE -> {
                    updateStatus(if (hasInternet) ConnectivityStatus.CONNECTED else ConnectivityStatus.NOT_CONNECTED)
                }
            }
        }
    }

    private fun debounceUpdate(action: suspend () -> Unit) {
        debounceJob?.cancel()
        debounceJob = GlobalScope.launch {
            delay(800) // 800ms debounce delay
            action()
        }
    }
    private fun checkInternetConnection(completion: (Boolean) -> Unit) {
        val url = "https://www.google.com/"
        GlobalScope.launch {
            try {
                val response: String = HttpClientProvider.client.get(url).toString()
                completion(true)
            } catch (e: Exception) {
                completion(false)
            }
        }
    }

    private fun updateStatus(newStatus: ConnectivityStatus) {
        _status.value = newStatus
    }

    actual companion object {
        private val _instance: ConnectivityMonitor by lazy { ConnectivityMonitor() }
        actual val instance: ConnectivityMonitor
            get() = _instance
    }
}
