/*
 * Decompiled with CFR 0.152.
 */
package network.arkane.provider.utils.crypto;

import java.math.BigInteger;
import java.util.Arrays;
import network.arkane.provider.BytesUtils;
import network.arkane.provider.utils.Assertions;
import network.arkane.provider.utils.CryptoUtils;
import network.arkane.provider.utils.crypto.ECDSASignature;
import network.arkane.provider.utils.crypto.ECKeyPair;
import org.bouncycastle.asn1.x9.X9IntegerConverter;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointCombMultiplier;
import org.bouncycastle.math.ec.custom.sec.SecP256K1Curve;

public class ECDSASign {
    public static SignatureData signMessage(byte[] message, ECKeyPair keyPair) throws Exception {
        return ECDSASign.signMessage(message, keyPair, true);
    }

    public static SignatureData signMessage(byte[] message, ECKeyPair keyPair, boolean needToHash) throws SignException {
        BigInteger publicKey = keyPair.getPublicKey();
        byte[] messageHash = needToHash ? CryptoUtils.blake2b(message) : message;
        int recId = -1;
        ECDSASignature sig = keyPair.sign(messageHash);
        for (int i = 0; i < 4; ++i) {
            BigInteger k = ECDSASign.recoverFromSignature(i, sig, messageHash);
            if (k == null || !k.equals(publicKey)) continue;
            recId = i;
            break;
        }
        if (recId == -1) {
            throw new SignException("Sign the data failed.");
        }
        if (recId == 2 || recId == 3) {
            throw new SignException("Recovery is not valid for VeChain MainNet.");
        }
        byte v = (byte)recId;
        byte[] r = BytesUtils.toBytesPadded((BigInteger)sig.r, (int)32);
        byte[] s = BytesUtils.toBytesPadded((BigInteger)sig.s, (int)32);
        return new SignatureData(v, r, s);
    }

    public static BigInteger recoverFromSignature(int recId, ECDSASignature sig, byte[] message) {
        Assertions.verifyPrecondition(recId == 0 || recId == 1, "recId must be 0 or 1");
        Assertions.verifyPrecondition(sig.r.signum() >= 0, "r must be positive");
        Assertions.verifyPrecondition(sig.s.signum() >= 0, "s must be positive");
        Assertions.verifyPrecondition(message != null, "message cannot be null");
        BigInteger n = ECKeyPair.CURVE.getN();
        BigInteger i = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = sig.r.add(i.multiply(n));
        BigInteger prime = SecP256K1Curve.q;
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        ECPoint R = ECDSASign.decompressKey(x, (recId & 1) == 1);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = new BigInteger(1, message);
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = sig.r.modInverse(n);
        BigInteger srInv = rInv.multiply(sig.s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint q = ECAlgorithms.sumOfTwoMultiplies((ECPoint)ECKeyPair.CURVE.getG(), (BigInteger)eInvrInv, (ECPoint)R, (BigInteger)srInv);
        byte[] qBytes = q.getEncoded(false);
        return new BigInteger(1, Arrays.copyOfRange(qBytes, 1, qBytes.length));
    }

    private static ECPoint decompressKey(BigInteger xBN, boolean yBit) {
        X9IntegerConverter x9 = new X9IntegerConverter();
        byte[] compEnc = x9.integerToBytes(xBN, 1 + x9.getByteLength(ECKeyPair.CURVE.getCurve()));
        compEnc[0] = (byte)(yBit ? 3 : 2);
        return ECKeyPair.CURVE.getCurve().decodePoint(compEnc);
    }

    public static BigInteger publicKeyFromPrivate(BigInteger privKey) {
        ECPoint point = ECDSASign.publicPointFromPrivate(privKey);
        byte[] encoded = point.getEncoded(false);
        return new BigInteger(1, Arrays.copyOfRange(encoded, 1, encoded.length));
    }

    private static ECPoint publicPointFromPrivate(BigInteger privKey) {
        if (privKey.bitLength() > ECKeyPair.CURVE.getN().bitLength()) {
            privKey = privKey.mod(ECKeyPair.CURVE.getN());
        }
        return new FixedPointCombMultiplier().multiply(ECKeyPair.CURVE.getG(), privKey);
    }

    public static class SignException
    extends RuntimeException {
        public SignException(String exceptionMessage) {
            super(exceptionMessage);
        }
    }

    public static class SignatureData {
        private final byte v;
        private final byte[] r;
        private final byte[] s;

        public SignatureData(byte v, byte[] r, byte[] s) {
            this.v = v;
            this.r = r;
            this.s = s;
        }

        public byte getV() {
            return this.v;
        }

        public byte[] getR() {
            return this.r;
        }

        public byte[] getS() {
            return this.s;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SignatureData that = (SignatureData)o;
            if (this.v != that.v) {
                return false;
            }
            if (!Arrays.equals(this.r, that.r)) {
                return false;
            }
            return Arrays.equals(this.s, that.s);
        }

        public byte[] toByteArray() {
            int size = this.r.length + this.s.length + 1;
            byte[] flat = new byte[size];
            System.arraycopy(this.r, 0, flat, 0, this.r.length);
            System.arraycopy(this.s, 0, flat, this.r.length, this.s.length);
            flat[size - 1] = this.v;
            return flat;
        }
    }
}

