(ns org.hellodata.api.oauth
  "Functions for generating ring-style oauth requests for the
  hellodata.org spa-api. These deliberately leave out sending the
  requests - choose whatever client is convenient - clj-http and
  http-kit should work"
  (:require [oauth.client]
            [oauth.signature]
            [org.hellodata.api.request :as request]))


(defn request-uri
  [{:keys [endpoint-url] :as config}]
  (str endpoint-url "/oauth/request_token"))

(defn access-uri
  [{:keys [endpoint-url] :as config}]
  (str endpoint-url "/oauth/access_token"))

(defn authorize-uri
  [{:keys [endpoint-url] :as config}]
  (str endpoint-url "/oauth/authorize"))

(defn consumer
  [{:keys [spa-identifier shared-secret] :as config}]
  {:pre [spa-identifier shared-secret]}
  (oauth.client/make-consumer spa-identifier shared-secret
                              (request-uri config)
                              (access-uri config)
                              (authorize-uri config)
                              :hmac-sha1))

(defn credentials
  [config token request-method url & params]
  (apply oauth.client/credentials
         (consumer config)
         (:oauth_token token)
         (:oauth_token_secret token)
         request-method
         url
         params))

(defn authorization-request
  "Build a ring-formatted oauth request for requesting a token"
  ([config]
   (authorization-request config (request-uri config)))
  ([config uri]
   (let [unsigned-params (-> (oauth.signature/oauth-params (consumer config)
                                                           (oauth.signature/rand-str 30)
                                                           (oauth.signature/msecs->secs (System/currentTimeMillis)))
                             (assoc :oauth_callback (:callback-url config)))]
     (-> (oauth.client/build-oauth-token-request (consumer config)
                                                 uri
                                                 unsigned-params
                                                 nil)
         request/accept-api-v1
         (assoc :url uri
                :request-method :post
                :throw-exceptions false)))))


(defn oauth-token?
  [{:keys [oauth_token oauth_token_secret]}]
  (boolean (and oauth_token oauth_token_secret)))

(defn response->token
  [{:keys [status body] :as oauth-token-response}]
  {:post [(oauth-token? %)]}
  (when (or (< status 200)
            (>= status 300))
    (throw (ex-info "Non-successful response" oauth-token-response)))
  (oauth.client/form-decode body))


(defn access-token-request
  "Get access token for user-approved request-token. `verifier` is the value
  that was provided by the redirect url by the user's browser after
  clicking though the `auth-approval-url`"
  [config request-token verifier]
  {:pre [(oauth-token? request-token)]}
  (let [unsigned-oauth-params (oauth.signature/oauth-params (consumer config)
                                                            (oauth.signature/rand-str 30)
                                                            (oauth.signature/msecs->secs (System/currentTimeMillis))
                                                            (:oauth_token request-token)
                                                            verifier)
        token-secret (:oauth_token_secret request-token)]
    (-> (oauth.client/build-oauth-token-request (consumer config)
                                                (access-uri config)
                                                unsigned-oauth-params
                                                nil
                                                token-secret)
        request/accept-api-v1
        (assoc :url (access-uri config)
               :request-method :post
               :throw-exceptions false))))


(defn authorized-request
  "Build an ring-formatted oauth request given an access-token"
  [config token request-method url]
  {:pre [(oauth-token? token)]}
  (-> (credentials config token request-method url)
      oauth.client/build-request
      (request/accept-api-v1)
      (assoc :url url
             :request-method request-method
             :throw-exceptions false)))

(defn approval-url
  "The url for the user to approve access to requested authorization"
  [config request-token]
  {:pre [(oauth-token? request-token)]}
  (oauth.client/user-approval-uri (consumer config) request-token))
