package org.monetdb.monetdbe;

import java.sql.*;
import java.util.ArrayList;
import java.util.List;

/**
 * A {@link Statement} suitable for the MonetDB embedded database.
 *
 * The object used for executing a static SQL statement and returning
 * the results it produces.
 *
 * By default, only one {@link ResultSet} object per Statement object can be
 * open at the same time. Therefore, if the reading of one ResultSet
 * object is interleaved with the reading of another, each must have
 * been generated by different {@link Statement} objects. All execution methods
 * in the Statement interface implicitly close a Statement's current
 * ResultSet object if an open one exists.
 */
public class MonetStatement extends MonetWrapper implements Statement {
    /** This statement's parent connection object */
    protected MonetConnection conn;
    /** The current result set */
    protected MonetResultSet resultSet;
    /** Batched SQL queries to execute */
    protected List<String> batch;
    /** Current updateCount (number of affected rows) */
    protected int updateCount = -1;
    /** Current updateCount (number of affected rows) for executeLarge methods  */
    protected long largeUpdateCount = -1;
    /** Timeout before aborting a query execution */
    private int queryTimeout = 0;
    /** Is this statement closed? */
    private boolean closed = false;
    /** Should this statement be closed when it's result set object is? */
    private boolean closeOnCompletion = false;
    /** The stack of warnings for this Statement object */
    private SQLWarning warnings;
    /** Maximum number of rows to fetch for resultset */
    private int maxRows = 0;
    /** Maximum number of rows to fetch for resultset for executeLarge methods */
    //This one isn't used right now
    private long largeMaxRows = 0;
    //These ones are ignored
    private int fetchSize;
    private int fetchDirection = ResultSet.FETCH_UNKNOWN;
    private int resultSetType = ResultSet.TYPE_SCROLL_INSENSITIVE;
    private int resultSetConcurrency = ResultSet.CONCUR_READ_ONLY;
    private int resultSetHoldability = ResultSet.HOLD_CURSORS_OVER_COMMIT;

    /**
     * Default MonetStatement constructor.
     *
     * @param conn the connection that created this Statement
     */
    public MonetStatement(MonetConnection conn) {
        this.conn = conn;
        this.resultSet = null;
        this.batch = new ArrayList<>();
    }

    /**
     * MonetStatement constructor with query timeout value.
     *
     * @param conn the connection that created this Statement
     * @param queryTimeout the timeout before aborting a query execution
     */
    public MonetStatement(MonetConnection conn, int queryTimeout) {
        this.conn = conn;
        this.resultSet = null;
        this.batch = new ArrayList<>();
        this.queryTimeout = queryTimeout;
    }

    /**
     * MonetStatement constructor with parameterized result set options.
     *
     * @param conn the connection that created this Statement
     * @param resultSetType type of ResultSet to produce
     * @param resultSetConcurrency concurrency of ResultSet to produce
     * @param resultSetHoldability holdability of ResultSet after commit
     */
    public MonetStatement(MonetConnection conn, int resultSetType, int resultSetConcurrency, int resultSetHoldability) {
        this.conn = conn;
        this.resultSet = null;
        this.resultSetType = resultSetType;
        this.resultSetConcurrency = resultSetConcurrency;
        this.resultSetHoldability = resultSetHoldability;
        this.batch = new ArrayList<>();
    }

    private final void addWarning(final String reason, final String sqlstate) {
        final SQLWarning warn = new SQLWarning(reason, sqlstate);
        if (warnings == null) {
            warnings = warn;
        } else {
            warnings.setNextWarning(warn);
        }
    }

    /**
     * Helper method to test whether the Statement object is closed
     * When closed, it throws an SQLException
     *
     * @throws SQLException if this method is called on a closed Statement
     */
    public void checkNotClosed() throws SQLException {
        if (isClosed())
            throw new SQLException("Statement is closed", "M1M20");
    }

    //Close
    /**
     * Releases this Statement object's database and JDBC resources immediately
     * instead of waiting for this to happen when it is automatically closed.
     *
     * Calling the method close on a Statement object that is already closed has
     * no effect.
     *
     * A Statement object is automatically closed when it is garbage collected.
     * When a Statement object is closed, its current ResultSet object, if one
     * exists, is also closed.
     *
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public void close() throws SQLException {
        if (isClosed())
            return;
        //If there is a result set and it is not closed, close it
        if (resultSet != null && !resultSet.isClosed()) {
            resultSet.close();
        }
        //If called on a PreparedStatement object, it should free the prepared statement in the database
        if (this instanceof MonetPreparedStatement) {
            //This check is here in case the Prepared Statement was not successful in the prepare stage
            if (((MonetPreparedStatement) this).statementNative != null) {
                MonetNative.monetdbe_cleanup_statement(conn.getDbNative(),((MonetPreparedStatement) this).statementNative);
            }
        }
        this.closed = true;
    }

    /**
     * This feature is not currently supported.
     *
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public void cancel() throws SQLException {
        throw new SQLFeatureNotSupportedException("cancel()");
    }

    /**
     * Retrieves whether this Statement object has been closed. A
     * Statement is closed if the method close has been called on it, or
     * if it is automatically closed.
     *
     * @return true if this Statement object is closed; false if it is
     *         still open
     */
    @Override
    public boolean isClosed() throws SQLException {
        return closed;
    }

    /**
     * Specifies that this Statement will be closed when all its
     * dependent result sets are closed.  If execution of the Statement
     * does not produce any result sets, this method has no effect.
     *
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public void closeOnCompletion() throws SQLException {
        checkNotClosed();
        closeOnCompletion = true;
    }

    /**
     * Returns a value indicating whether this Statement will be closed
     * when all its dependent result sets are closed.
     *
     * @return true if the Statement will be closed when all of its
     *         dependent result sets are closed; false otherwise
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public boolean isCloseOnCompletion() throws SQLException {
        checkNotClosed();
        return closeOnCompletion;
    }

    /**
     * Called by the result set object when it is closed
     * Used to close statement if closeOnCompletion() was called
     *
     * @throws SQLException if this method is called on a closed Statement
     */
    protected void closeIfComplete () throws SQLException {
        if (!closed && closeOnCompletion) {
            close();
        }
    }

    /**
     * The execute method executes the given SQL statement and indicates the form of
     * the first result. You must then use the methods getResultSet or
     * getUpdateCount to retrieve the result (either a ResultSet object or an int representing the update count).
     *
     * Multiple results may result from the SQL statement, but only the first one may be retrieved in the current version.
     *
     * The result set or update count object variables are set within monetdbe_query().
     *
     * @param sql any SQL statement
     * @return true if the first result is a ResultSet object; false if it is an
     *         update count or there are no results
     * @throws SQLException if a database access error occurs or this method is called on a closed Statement
     */
    @Override
    public boolean execute(String sql) throws SQLException {
        checkNotClosed();

        int lastUpdateCount = this.updateCount;
        MonetResultSet lastResultSet = this.resultSet;
        this.resultSet = null;
        this.updateCount = -1;

        //ResultSet and UpdateCount is set within monetdbe_query
        String error_msg = MonetNative.monetdbe_query(conn.getDbNative(),sql,this,false, getMaxRows());
        if (error_msg != null) {
            this.updateCount = lastUpdateCount;
            this.resultSet = lastResultSet;
            throw new SQLException(error_msg);
        }
        else if (this.resultSet!=null) {
            return true;
        }
        //Data manipulation and data definition queries
        else if (this.updateCount >= 0){
            return false;
        }
        else {
            throw new SQLException("Error in monetdbe_query");
        }
    }

    /**
     * Executes the given SQL statement, which returns a single ResultSet
     * object.
     *
     * Multiple results may result from the SQL statement, but only the first one may be retrieved in the current version.
     *
     * @param sql an SQL statement to be sent to the database, typically a
     *        static SQL SELECT statement
     * @return a ResultSet object that contains the data produced by the given
     *         query; never null
     * @throws SQLException if a database access error occurs or the given SQL
     *         statement produces anything other than a single ResultSet object
     */
    @Override
    public ResultSet executeQuery(final String sql) throws SQLException {
        if (!execute(sql))
            throw new SQLException("Query did not produce a result set", "M1M19");
        return getResultSet();
    }

    /**
     * Executes the given SQL statement, which may be an INSERT, UPDATE, or
     * DELETE statement or an SQL statement that returns nothing, such as an
     * SQL DDL statement.
     *
     * @param sql an SQL INSERT, UPDATE or DELETE statement or an SQL statement
     *        that returns nothing
     * @return either the row count for INSERT, UPDATE  or DELETE statements, or
     *         0 for SQL statements that return nothing
     * @throws SQLException if a database access error occurs or the given SQL
     *         statement produces a ResultSet object
     */
    @Override
    public int executeUpdate(final String sql) throws SQLException {
        if (execute(sql))
            throw new SQLException("Query produced a result set", "M1M17");
        return getUpdateCount();
    }

    /**
     * Executes the given SQL statement, which may be an INSERT, UPDATE, or
     * DELETE statement or an SQL statement that returns nothing, such as an
     * SQL DDL statement.
     *
     * This method should be used when the returned row count may exceed Integer.MAX_VALUE.
     *
     * The result set or large update count object variables are set within monetdbe_query().
     *
     * @param sql an SQL Data Manipulation Language (DML) statement, such as
     *	INSERT, UPDATE or DELETE; or an SQL statement that returns nothing,
     *	such as a DDL statement.
     * @return either the row count for INSERT, UPDATE  or DELETE statements, or
     *          0 for SQL statements that return nothing
     * @throws SQLException if a database access error occurs, this method is
     *	called on a closed Statement, the given SQL statement produces a
     *	ResultSet object, the method is called on a PreparedStatement or CallableStatement
     */
    @Override
    public long executeLargeUpdate(String sql) throws SQLException {
        checkNotClosed();

        long lastUpdateCount = this.largeUpdateCount;
        MonetResultSet lastResultSet = this.resultSet;
        this.resultSet = null;
        this.largeUpdateCount = -1;

        //ResultSet and UpdateCount is set within monetdbe_query
        String error_msg = MonetNative.monetdbe_query(conn.getDbNative(),sql,this, true, getMaxRows());
        if (error_msg != null) {
            this.largeUpdateCount = lastUpdateCount;
            this.resultSet = lastResultSet;
            throw new SQLException(error_msg);
        }
        else if (this.resultSet!=null) {
            throw new SQLException("Query produced a result set", "M1M17");
        }
        else {
            return getLargeUpdateCount();
        }
    }

    /**
     * Submits a batch of commands to the database for execution and if all commands execute successfully,
     * returns an array of update counts. The int elements of the array that is returned are ordered to correspond
     * to the commands in the batch, which are ordered according to the order in which they were added to the batch.
     *
     * The elements in the array returned by the method executeBatch may be one of the following:
     * <ol>
     *     <li>A number greater than or equal to zero -- indicates that the command was processed successfull and is an
     *     update count giving the number of rows in the database that were affected by the command's execution </li>
     *     <li>A value of SUCCESS_NO_INFO -- indicates that the command was processed successfully
     *     but that the number of rows affected is unknown </li>
     * </ol>
     *
     * @return an array of update counts containing one element for each command in the batch.
     * The elements of the array are ordered according to the order in which commands were added to the batch.
     * @throws SQLException if a database access error occurs or this method is called on a closed Statement
     * @throws BatchUpdateException if one of the commands sent to the database fails to execute properly or attempts to return a result set
     */
    @Override
    public int[] executeBatch() throws SQLException {
        checkNotClosed();
        if (batch == null || batch.isEmpty()) {
            return new int[0];
        }
        long[] largeCounts = executeLargeBatch();

        //Copy from long[] to int[]
        int[] counts = new int[largeCounts.length];
        for (int i = 0; i < largeCounts.length; i++)
            counts[i] = (largeCounts[i] >= Integer.MAX_VALUE) ? Integer.MAX_VALUE : (int)largeCounts[i];
        return counts;
    }

    /**
     * Submits a batch of commands to the database for execution and if all commands execute successfully,
     * returns an array of update counts. The long elements of the array that is returned are ordered to correspond
     * to the commands in the batch, which are ordered according to the order in which they were added to the batch.
     *
     * This method should be used when the returned row count may exceed Integer.MAX_VALUE.
     *
     * @return an array of update counts containing one element for each command in the batch.
     * The elements of the array are ordered according to the order in which commands were added to the batch.
     * @throws SQLException if a database access error occurs or this method is called on a closed Statement
     * @throws BatchUpdateException if one of the commands sent to the database fails to execute properly or attempts to return a result set
     */
    @Override
    public long[] executeLargeBatch() throws SQLException {
        checkNotClosed();
        if (batch == null || batch.isEmpty()) {
            return new long[0];
        }
        long[] counts = new long[batch.size()];
        long count = -1;

        for (int i = 0; i < batch.size(); i++) {
            String query = batch.get(i);
            try {
                count = executeLargeUpdate(query);
            } catch (SQLException e) {
                //Query returned a resultSet or query failed, throw BatchUpdateException
                throw new BatchUpdateException();
            }
            if (count >= 0) {
                counts[i] = count;
            }
            else {
                counts[i] = Statement.SUCCESS_NO_INFO;
            }
        }
        clearBatch();
        return counts;
    }

    /**
     * Adds the given SQL command to the current list of commands for this
     * Statement object.  The commands in this list can be executed as a
     * batch by calling the method executeBatch.
     *
     * @param sql typically this is a SQL INSERT or UPDATE statement
     * @throws SQLException if a database access error occurs, this method is called on a closed Statement,
     * the driver does not support batch updatesor the method is called on a PreparedStatement or CallableStatement
     */
    @Override
    public void addBatch(String sql) throws SQLException {
        checkNotClosed();
        if (batch == null) {
            batch = new ArrayList<>();
        }
        batch.add(sql);
    }

    /**
     * Empties this Statement object's current list of SQL commands.
     * @throws  SQLException if a database access error occurs, this method is called on a closed Statement
     * or the driver does not support batch updates
     */
    @Override
    public void clearBatch() throws SQLException {
        checkNotClosed();
        if (batch != null) {
            batch.clear();
        }
    }

    /**
     * Feature currently not supported.
     *
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public boolean getMoreResults(int current) throws SQLException {
        //TODO GETMORERESULTS
        throw new SQLFeatureNotSupportedException("getMoreResults()");
    }

    /**
     * Feature currently not supported.
     *
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public boolean getMoreResults() throws SQLException {
        //TODO GETMORERESULTS
        throw new SQLFeatureNotSupportedException("getMoreResults()");
    }

    /**
     * Retrieves the current result as an update count; if the result is a ResultSet object or there are no more results,
     * -1 is returned. This method should be called only once per result.
     *
     * @return the current result as an update count; -1 if the current result is a ResultSet object or there are no more results
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getUpdateCount() throws SQLException {
        checkNotClosed();
        int update=-1;
        //We should only return the update count on the first call
        if (this.updateCount != -1) {
            update = this.updateCount;
            this.updateCount = -1;
        }
        return update;
    }

    /**
     * Retrieves the current result as a ResultSet object. This method should be called only once per result.
     *
     * @return the current result as a ResultSet object or null if the result is an update count or there are no more results
     * @throws SQLException if  this method is called on a closed Statement
     */
    @Override
    public ResultSet getResultSet() throws SQLException {
        checkNotClosed();
        return resultSet;
    }

    /**
     * Sets escape processing on or off.
     * This feature is not supported by this driver, so calling this method will add a SQL warning.
     *
     * @param enable true to enable escape processing; false to disable it
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
        checkNotClosed();
        if (enable)
            addWarning("setEscapeProcessing: JDBC escape syntax is not supported by this driver", "01M22");
    }

    /**
     * Retrieves the number of seconds the driver will wait for a Statement object to execute.
     * If the limit is exceeded, a SQLException is thrown.
     *
     * The current version of the driver does not support query timeout, so this limit will have no effect on query execution.
     *
     * @return the current query timeout limit in seconds; zero means there is no limit
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getQueryTimeout() throws SQLException {
        checkNotClosed();
        return queryTimeout;
    }

    /**
     * Sets the number of seconds the driver will wait for a Statement object to execute to the given number of seconds.
     * By default there is no limit on the amount of time allowed for a running statement to complete.
     * If the limit is exceeded, an SQLTimeoutException is thrown.
     *
     * The driver must apply this limit to the execute, executeQuery and executeUpdate methods.
     *
     * The current version of the driver does not support query timeout, so this limit will have no effect on query execution.
     *
     * @param seconds the new query timeout limit in seconds; zero means there is no limit
     * @throws SQLException if this method is called on a closed Statement or the condition seconds &gt;= 0 is not satisfied
     */
    @Override
    public void setQueryTimeout(int seconds) throws SQLException {
        checkNotClosed();
        if (seconds < 0)
            throw new SQLException("Illegal timeout value: " + seconds);
        queryTimeout = seconds;
    }

    /**
     * Retrieves the first warning reported by calls on this Statement
     * object.  If there is more than one warning, subsequent warnings
     * will be chained to the first one and can be retrieved by calling
     * the method SQLWarning.getNextWarning on the warning that was
     * retrieved previously.
     *
     * @return the first SQLWarning object or null if there are none
     * @throws SQLException if a database access error occurs or this method is
     *         called on a closed connection
     */
    @Override
    public SQLWarning getWarnings() throws SQLException {
        checkNotClosed();
        return warnings;
    }

    /**
     * Clears all warnings reported for this Statement object. After a
     * call to this method, the method getWarnings returns null until a
     * new warning is reported for this Connection object.
     *
     * @throws SQLException if a database access error occurs or this method is called on a closed connection
     */
    @Override
    public void clearWarnings() throws SQLException {
        checkNotClosed();
        warnings = null;
    }

    /**
     * Sets the SQL cursor name to the given String, which will be used by subsequent Statement object execute methods.
     *
     * Because positioned updates/deletes are not supported by this driver, this will add a SQL warning.
     *
     * @param name the new cursor name, which must be unique within a connection
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public void setCursorName(String name) throws SQLException {
        checkNotClosed();
        addWarning("setCursorName: positioned updates/deletes not supported", "01M21");
    }

    /**
     * Gives the driver a hint as to the direction in which rows will be processed in ResultSet objects
     * created using this Statement object. The default value is ResultSet.FETCH_FORWARD.
     *
     * @param direction the initial direction for processing rows
     * @throws SQLException if this method is called on a closed Statement or the given direction is not one of
     * ResultSet.FETCH_FORWARD, ResultSet.FETCH_REVERSE, or ResultSet.FETCH_UNKNOWN
     */
    @Override
    public void setFetchDirection(int direction) throws SQLException {
        checkNotClosed();
        if (direction == ResultSet.FETCH_FORWARD ||
                direction == ResultSet.FETCH_REVERSE ||
                direction == ResultSet.FETCH_UNKNOWN)
        {
            fetchDirection = direction;
        } else {
            throw new SQLException("Illegal direction: " + direction, "M1M05");
        }
    }

    /**
     * Retrieves the direction for fetching rows from database tables that is the default for result sets generated from this Statement object.
     *
     * @return the default fetch direction for result sets generated from this Statement object
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getFetchDirection() throws SQLException {
        checkNotClosed();
        return fetchDirection;
    }

    /**
     * Gives the JDBC driver a hint as to the number of rows that should be fetched from the database when more rows are
     * needed for ResultSet objects generated by this Statement.
     *
     * If the value specified is zero, then the hint is ignored. The default value is zero.
     *
     * @param rows the number of rows to fetch
     * @throws SQLException if this method is called on a closed Statement or the condition rows &gt;= 0 is not satisfied
     */
    @Override
    public void setFetchSize(int rows) throws SQLException {
        checkNotClosed();
        if (rows >= 0 && !(getMaxRows() != 0 && rows > getMaxRows())) {
            this.fetchSize = rows;
        } else {
            throw new SQLException("Illegal fetch size value: " + rows, "M1M05");
        }
    }

    /**
     * Retrieves the number of result set rows that is the default fetch size for ResultSet objects generated from this Statement object.
     *
     * @return the default fetch size for result sets generated from this Statement object
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getFetchSize() throws SQLException {
        checkNotClosed();
        return fetchSize;
    }

    /**
     * Retrieves the result set concurrency for ResultSet objects generated by this Statement object.
     *
     * @return either ResultSet.CONCUR_READ_ONLY or ResultSet.CONCUR_UPDATABLE
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getResultSetConcurrency() throws SQLException {
        checkNotClosed();
        return resultSetConcurrency;
    }

    /**
     * Retrieves the result set type for ResultSet objects generated by this Statement object.
     *
     * @return one of ResultSet.TYPE_FORWARD_ONLY, ResultSet.TYPE_SCROLL_INSENSITIVE, or ResultSet.TYPE_SCROLL_SENSITIVE
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getResultSetType() throws SQLException {
        checkNotClosed();
        return resultSetType;
    }

    /**
     * Retrieves the Connection object that produced this Statement object.
     *
     * @return the connection that produced this statement
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public Connection getConnection() throws SQLException {
        checkNotClosed();
        return conn;
    }

    /**
     * Retrieves the result set holdability for ResultSet objects generated by this Statement object.
     *
     * @return either ResultSet.HOLD_CURSORS_OVER_COMMIT or ResultSet.CLOSE_CURSORS_AT_COMMIT
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getResultSetHoldability() throws SQLException {
        checkNotClosed();
        return resultSetHoldability;
    }

    /**
     * Feature not supported
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public void setPoolable(boolean poolable) throws SQLException {
        throw new SQLFeatureNotSupportedException("Poolable statements are currently not supported by the driver.");
    }

    /**
     * Returns a value indicating whether the Statement is poolable or not.
     *
     * @return false
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public boolean isPoolable() throws SQLException {
        checkNotClosed();
        return false;
    }

    /**
     * Retrieves the current result as an update count; if the result is a ResultSet object or there are no more results,
     * -1 is returned. This method should be called only once per result.
     *
     * This method should be used when the returned row count may exceed Integer.MAX_VALUE.
     *
     * @return the current result as an update count; -1 if the current result is a ResultSet object or there are no more results
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public long getLargeUpdateCount() throws SQLException {
        checkNotClosed();
        long update=-1;
        //We should only return the update count on the first call
        if (this.largeUpdateCount != -1) {
            update = this.largeUpdateCount;
            this.largeUpdateCount = -1;
        }
        return update;
    }

    /**
     * Sets the limit for the maximum number of rows that any ResultSet object generated by this Statement object can
     * contain to the given number. If the limit is exceeded, the excess rows are silently dropped.
     *
     * This method should be used when the row limit may exceed Integer.MAX_VALUE.
     *
     * @param max the new max rows limit; zero means there is no limit
     * @throws SQLException if this method is called on a closed Statement or the condition max &gt;= 0 is not satisfied
     */
    @Override
    public void setLargeMaxRows(long max) throws SQLException {
        checkNotClosed();
        this.largeMaxRows = max;
    }

    /**
     * Retrieves the maximum number of rows that a ResultSet object produced by this Statement object can contain.
     * If this limit is exceeded, the excess rows are silently dropped.
     *
     * This method should be used when the returned row limit may exceed Integer.MAX_VALUE.
     *
     * @return the current maximum number of rows for a ResultSet object produced by this Statement object; zero means there is no limit
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public long getLargeMaxRows() throws SQLException {
        checkNotClosed();
        return largeMaxRows;
    }

    /**
     * Retrieves the maximum number of bytes that can be returned for character and binary column values in a
     * ResultSet object produced by this Statement object.
     *
     * @return zero, meaning there is no column size limit for columns storing character and binary values
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getMaxFieldSize() throws SQLException {
        checkNotClosed();
        // MonetDB supports null terminated strings of max 2GB, see function: int UTF8_strlen();
        //return 2*1024*1024*1024 - 2;
        return 0;
    }

    /**
     * Sets the limit for the maximum number of bytes that can be returned for character and binary column values in a
     * ResultSet object produced by this Statement object.
     *
     * @param max the new column size limit in bytes; zero means there is no limit
     * @throws SQLException if this method is called on a closed Statement or the condition max &gt;= 0 is not satisfied
     */
    @Override
    public void setMaxFieldSize(final int max) throws SQLException {
        checkNotClosed();
        if (max < 0)
            throw new SQLException("Illegal max value: " + max, "M1M05");
        if (max > 0)
            addWarning("setMaxFieldSize: field size limitation not supported", "01M23");
    }

    /**
     * Retrieves the maximum number of rows that a ResultSet object produced by this Statement object can contain.
     * If this limit is exceeded, the excess rows are silently dropped.
     *
     * @return the current maximum number of rows for a ResultSet object produced by this Statement object; zero means there is no limit
     * @throws SQLException if this method is called on a closed Statement
     */
    @Override
    public int getMaxRows() throws SQLException {
        checkNotClosed();
        return maxRows;
    }

    /**
     * Sets the limit for the maximum number of rows that any ResultSet object generated by this Statement object can contain to the given number.
     * If the limit is exceeded, the excess rows are silently dropped.
     *
     * @param max the new max rows limit; zero means there is no limit
     * @throws SQLException if this method is called on a closed Statement or the condition max &gt;= 0 is not satisfied
     */
    @Override
    public void setMaxRows(int max) throws SQLException {
        checkNotClosed();
        if (max < 0)
            throw new SQLException("Illegal max value: " + max, "M1M05");
        maxRows = max;
    }

    //TODO Generated Keys

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        return false;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        return false;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        return false;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        return 0;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        return 0;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        return 0;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public long executeLargeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        return 0;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public long executeLargeUpdate(String sql, int[] columnIndexes) throws SQLException {
        return 0;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public long executeLargeUpdate(String sql, String[] columnNames) throws SQLException {
        return 0;
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported
     */
    @Override
    public ResultSet getGeneratedKeys() throws SQLException {
        return null;
    }
}
