package org.monetdb.monetdbe;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.*;
import java.sql.*;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeParseException;
import java.util.Calendar;
import java.util.Map;

/**
 * A {@link ResultSet} suitable for the MonetDB embedded database.
 *
 * A table of data representing a database result set, which is usually generated by executing a statement that queries the database.
 * A ResultSet object maintains a cursor pointing to its current row of data. Initially the cursor is positioned before the first row.
 * The next method moves the cursor to the next row, and because it returns false when there are no more rows in the ResultSet object,
 * it can be used in a while loop to iterate through the result set.
 *
 * The ResultSet interface provides getter methods (getBoolean, getLong, and so on) for retrieving column values from the current row.
 * Values can be retrieved using either the index number of the column or the name of the column.
 * In general, using the column index will be more efficient. Columns are numbered from 1.
 */
public class MonetResultSet extends MonetWrapper implements ResultSet {
    /** This result set's parent statement object */
    private final MonetStatement statement;
    /** The pointer to the C result set object */
    private ByteBuffer nativeResult;
    /** Metadata object containing info about this result set */
    private MonetResultSetMetaData metaData;
    /** Number of rows in result */
    private final int nRows;
    /** Number of columns in result */
    private final int nColumns;
    /** Current cursor position */
    private int curRow;
    /** Result columns */
    private MonetColumn[] columns;
    /** Result set name */
    private String name;
    /** The stack of warnings for this Statement object */
    private SQLWarning warnings;
    /** Whether the last read from results was a null */
    private boolean lastReadWasNull = true;
    /** Whether this result set is closed */
    private boolean closed = false;

    //Ignored
    private int resultSetType = ResultSet.TYPE_SCROLL_INSENSITIVE;
    private int concurrency = ResultSet.CONCUR_READ_ONLY;
    private int fetchDirection = ResultSet.FETCH_UNKNOWN;
    private int resultSetHoldability = ResultSet.HOLD_CURSORS_OVER_COMMIT;
    private int fetchSize;

    /**
     * Result constructor, called from native function returnResult().
     *
     * @param statement Parent statement object
     * @param nativeResult Pointer to C result set
     * @param nRows Number of rows
     * @param nColumns Number of columns
     * @param name Result name
     * @param maxRows Maximum rows, set by the parent statement object
     */
    MonetResultSet(MonetStatement statement, ByteBuffer nativeResult, int nRows, int nColumns, String name, int maxRows) {
        this.statement = statement;
        this.nativeResult = nativeResult;
        this.nColumns = nColumns;
        this.curRow = 0;
        this.columns = MonetNative.monetdbe_result_fetch_all(nativeResult,nRows,nColumns);

        //Failed fetch, destroy resultset
        if (this.columns == null) {
            System.out.println("ResultSet fetch error");
            try {
                this.close();
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }

        this.name = name;

        if (maxRows != 0 && maxRows < nRows) {
            this.nRows = maxRows;
        }
        else {
            this.nRows = nRows;
        }
    }

    public int getRowsNumber() {
        return nRows;
    }

    public int getColumnsNumber() {
        return nColumns;
    }

    /**
     * Gets the value of the designated column in the current row as an Java object.
     * The type of the Java object will be the default Java object type corresponding to the column's SQL type,
     * following the mapping for built-in types specified in the JDBC specification.
     * If the value is an SQL NULL, the driver returns a Java null.
     *
     * @param columnIndex Column index (starts at 1)
     * @return a java.lang.Object holding the column value
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Object getObject(int columnIndex) throws SQLException {
        checkNotClosed();
        if (columnIndex > nColumns) {
            throw new SQLException("columnIndex is not valid");
        }
        int type = columns[columnIndex-1].getMonetdbeType();
        switch (type) {
            case 0:
                return getBoolean(columnIndex);
            case 1:
                return getShort(columnIndex);
            case 2:
                return getShort(columnIndex);
            case 3:
                return getInt(columnIndex);
            case 4:
                return getLong(columnIndex);
            case 5:
                return getBigInteger(columnIndex);
            case 6:
                return getInt(columnIndex);
            case 7:
                return getFloat(columnIndex);
            case 8:
                return getDouble(columnIndex);
            case 9:
                return getString(columnIndex);
            case 10:
                return getBlob(columnIndex);
            case 11:
                return getDate(columnIndex);
            case 12:
                return getTime(columnIndex);
            case 13:
                return getTimestamp(columnIndex);
            default:
                return null;
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as an Java object.
     * This method uses the specified Map object for custom mapping if appropriate.
     * If there is no mapping for the column's SQL type, return the default java object type for that column.
     *
     * @param columnIndex Column index (starts at 1)
     * @param map a java.util.Map object that contains the mapping from SQL type names to classes in the Java programming language
     * @return an Object representing the SQL value in the specified column
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Object getObject(int columnIndex, Map<String, Class<?>> map) throws SQLException {
        checkNotClosed();
        if (columnIndex > nColumns) {
            throw new SQLException("columnIndex is not valid");
        }
        else if (map == null) {
            //If there is no mapping, return default Java class
            return getObject(columnIndex);
        }

        int monetdbeType = columns[columnIndex-1].getMonetdbeType();
        String sqlDefaultType = MonetTypes.getSQLTypeNameFromMonet(monetdbeType);
        Class<?> convertClass;

        if (sqlDefaultType.equals("NULL")) {
            return null;
        }

        //Map contains mapping to Java class from default SQL type for the column's monetdbeType
        if (map.containsKey(sqlDefaultType)) {
            convertClass = map.get(sqlDefaultType);
            return getObject(columnIndex,convertClass);
        }
        //Alternative SQL types for the column's monetdbeType
        else {
            for (String keyType : map.keySet()) {
                if (MonetTypes.getMonetTypeIntFromSQLName(keyType) == monetdbeType) {
                    convertClass = map.get(keyType);
                    return getObject(columnIndex,convertClass);
                }
            }
        }
        //If there is no possible mapping between the monetdbetype and a SQL type in the argument map, return default Java class
        return getObject(columnIndex);
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object and will convert from
     * the SQL type of the column to the requested Java data type, if the conversion is supported.
     * If the conversion is not supported or null is specified for the type, a SQLException is thrown.
     *
     * @param columnIndex Column index (starts at 1)
     * @param type Class representing the Java data type to convert the designated column to
     * @return an instance of type holding the column value
     * @throws SQLException  if conversion is not supported, type is null or another error occurs
     */
    //TODO: This object conversion probably doesn't work well. We can't simply cast from the default Java object associated with the monetdbetype.
    @Override
    public <T> T getObject(int columnIndex, Class<T> type) throws SQLException {
        checkNotClosed();
        if (type == null) {
            throw new SQLException("Type is null");
        }
        else if (columnIndex > nColumns) {
            throw new SQLException("columnIndex is not valid");
        }

        int monetdbeType = columns[columnIndex-1].getMonetdbeType();
        if (MonetTypes.convertTojavaClass(monetdbeType,type)) {
            Object defaultValue = getObject(columnIndex);
            return type.cast(defaultValue);
        }
        else {
            throw new SQLException("Conversion is not supported");
        }
    }

    //Gets
    /**
     * Maps the given ResultSet column label to its ResultSet column index.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @return the column index of the given column name
     * @throws SQLException if the ResultSet object does not contain a column labeled columnLabel or this method is called on a closed result
     */
    @Override
    public int findColumn(String columnLabel) throws SQLException {
        checkNotClosed();
        String[] names = ((MonetResultSetMetaData)this.getMetaData()).getNames();
        if (columnLabel != null) {
            final int array_size = names.length;
            for (int i = 0; i < array_size; i++) {
                if (columnLabel.equals(names[i]))
                    return i + 1;
            }
            /* if an exact match did not succeed try a case insensitive match */
            for (int i = 0; i < array_size; i++) {
                if (columnLabel.equalsIgnoreCase(names[i]))
                    return i + 1;
            }
        }
        throw new SQLException("No such column name: " + columnLabel, "M1M05");
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a String in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public String getString(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            String val = columns[columnIndex-1].getString(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a boolean in the Java programming language.
     *
     * If the designated column has a datatype of CHAR or VARCHAR and contains a "0"
     * or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT and contains a 0,
     * a value of false is returned. If the designated column has a datatype of CHAR or VARCHAR
     * and contains a "1" or has a datatype of BIT, TINYINT, SMALLINT, INTEGER or BIGINT and contains a 1,
     * a value of true is returned.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is false
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public boolean getBoolean(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            Boolean val = columns[columnIndex-1].getBoolean(curRow-1);
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a byte in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is 0
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public byte getByte(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            Byte val = columns[columnIndex-1].getByte(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return 0;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a short in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is 0
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public short getShort(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            Short val = columns[columnIndex-1].getShort(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return 0;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a int in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is 0
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public int getInt(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            Integer val = columns[columnIndex-1].getInt(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return 0;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a long in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is 0
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public long getLong(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            Long val = columns[columnIndex-1].getLong(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return 0;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a float in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is 0
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public float getFloat(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            Float val = columns[columnIndex-1].getFloat(curRow-1);
            if (val.isNaN()) {
                lastReadWasNull = true;
                return 0;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a double in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is 0
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public double getDouble(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            Double val = columns[columnIndex-1].getDouble(curRow-1);
            if (val.isNaN()) {
                lastReadWasNull = true;
                return 0;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a byte array in the Java programming language.
     * The bytes represent the raw values returned by the driver.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public byte[] getBytes(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            byte[] val = columns[columnIndex-1].getBytes(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.sql.BigDecimal in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public BigDecimal getBigDecimal(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            BigDecimal val = columns[columnIndex-1].getBigDecimal(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return BigDecimal.ZERO;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.sql.BigInteger in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    public BigInteger getBigInteger (int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            BigInteger val = columns[columnIndex-1].getBigInteger(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return BigInteger.ZERO;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.time.LocalDate in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid, the LocalDate value fetched from the database cannot be parsed
     * or this method is called on a closed result set
     */
    public LocalDate getLocalDate(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            LocalDate val = columns[columnIndex-1].getLocalDate(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        } catch (DateTimeParseException e) {
            throw new SQLException("DateTime string could not be parsed");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.sql.Date in the Java programming language.
     * This method uses the given calendar to set the timezone.
     *
     * @param columnIndex Column index (starts at 1)
     * @param cal the java.util.Calendar object to use in constructing the date
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Date getDate(int columnIndex, Calendar cal) throws SQLException {
        LocalDate val = getLocalDate(columnIndex);
        //Set timezone if there is one
        if (cal != null && val != null) {
            val = LocalDateTime.of(val, LocalTime.now())
                    .atZone(cal.getTimeZone().toZoneId())
                    .withZoneSameInstant(ZoneOffset.UTC)
                    .toLocalDate();
        }
        return val != null ? Date.valueOf(val) : null;
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.time.LocalTime in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid, the LocalTime value fetched from the database cannot be parsed
     * or this method is called on a closed result set
     */
    public LocalTime getLocalTime(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            LocalTime val = columns[columnIndex-1].getLocalTime(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        } catch (DateTimeParseException e) {
            throw new SQLException("DateTime string could not be parsed");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.sql.Time in the Java programming language.
     * This method uses the given calendar to set the timezone.
     *
     * @param columnIndex Column index (starts at 1)
     * @param cal the java.util.Calendar object to use in constructing the date
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Time getTime(int columnIndex, Calendar cal) throws SQLException {
        //Get LocalTime without timezone
        LocalTime val = getLocalTime(columnIndex);
        //Set timezone if there is one
        if (cal != null && val != null) {
            val = LocalDateTime.of(LocalDate.now(), val)
                    .atZone(cal.getTimeZone().toZoneId())
                    .withZoneSameInstant(ZoneOffset.UTC)
                    .toLocalTime();
        }
        return val != null ? Time.valueOf(val) : null;
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.time.LocalDateTime in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid, the LocalDateTime value fetched from the database cannot be parsed
     * or this method is called on a closed result set
     */
    public LocalDateTime getLocalDateTime(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            LocalDateTime val = columns[columnIndex-1].getLocalDateTime(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        } catch (DateTimeParseException e) {
            throw new SQLException("DateTime string could not be parsed");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.sql.Timestamp in the Java programming language.
     * This method uses the given calendar to set the timezone.
     *
     * @param columnIndex Column index (starts at 1)
     * @param cal the java.util.Calendar object to use in constructing the date
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Timestamp getTimestamp(int columnIndex, Calendar cal) throws SQLException {
        LocalDateTime val = getLocalDateTime(columnIndex);
        //Set timezone if there is one
        if (cal != null && val != null) {
            val = val.atZone(cal.getTimeZone().toZoneId())
                    .withZoneSameInstant(ZoneOffset.UTC)
                    .toLocalDateTime();
        }
        return val != null ? Timestamp.valueOf(val) : null;
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a Blob object in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return a Blob object representing the SQL BLOB value in the specified column
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Blob getBlob(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            MonetBlob val = columns[columnIndex-1].getBlob(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return val;
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a Clob object in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return a Clob object representing the SQL CLOB value in the specified column
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Clob getClob(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            String val = columns[columnIndex-1].getString(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return new MonetClob(val);
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.net.URL object in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value as a java.net.URL object; if the value is SQL NULL, the value returned is null in the Java programming language
     * @throws SQLException if the columnIndex is not valid, this method is called on a closed result set or if a URL is malformed
     */
    @Override
    public URL getURL(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            String val = columns[columnIndex-1].getString(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return new URL(val);
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        } catch (MalformedURLException e) {
            throw new SQLException("column is not a valid URL");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a stream of uninterpreted bytes.
     * The value can then be read in chunks from the stream. This method is particularly suitable for retrieving large LONGVARBINARY values.
     *
     * @param columnIndex Column index (starts at 1)
     * @return a Java input stream that delivers the database column value as a stream of uninterpreted bytes; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public InputStream getBinaryStream(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            Blob val = columns[columnIndex-1].getBlob(curRow-1);
            if (val == null)
                return null;
            return val.getBinaryStream();
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.io.Reader object.
     *
     * @param columnIndex Column index (starts at 1)
     * @return a java.io.Reader object that contains the column value; if the value is SQL NULL, the value returned is null in the Java programming language.
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Reader getCharacterStream(int columnIndex) throws SQLException {
        checkNotClosed();
        if (curRow <= 0 || curRow > nRows)
            throw new SQLException("Current row " + curRow + " does not support operation");
        try {
            String val = columns[columnIndex-1].getString(curRow-1);
            if (val == null) {
                lastReadWasNull = true;
                return null;
            }
            lastReadWasNull = false;
            return new java.io.StringReader(val);
        } catch (IndexOutOfBoundsException e) {
            throw new SQLException("columnIndex out of bounds");
        }
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.sql.Date in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Date getDate(int columnIndex) throws SQLException {
        return getDate(columnIndex,null);
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.sql.Time in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Time getTime(int columnIndex) throws SQLException {
        return getTime(columnIndex,null);
    }

    /**
     * Retrieves the value of the designated column in the current row of this ResultSet object as a java.sql.Timestamp in the Java programming language.
     *
     * @param columnIndex Column index (starts at 1)
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     */
    @Override
    public Timestamp getTimestamp(int columnIndex) throws SQLException {
        return getTimestamp(columnIndex,null);
    }

    /**
     * Helper method to test whether the ResultSet object is closed
     * When closed, it throws an SQLException
     * @throws SQLException if the ResultSet object is closed
     */
    private void checkNotClosed() throws SQLException {
        if (isClosed())
            throw new SQLException("ResultSet is closed", "M1M20");
    }

    /**
     * Retrieves whether this ResultSet object has been closed. A
     * Statement is closed if the method close has been called on it, or
     * if it is automatically closed.
     *
     * @return true if this ResultSet object is closed; false if it is
     *         still open
     */
    @Override
    public boolean isClosed() throws SQLException {
        return this.closed;
    }

    /**
     * Releases this ResultSet object's database and JDBC resources immediately
     * instead of waiting for this to happen when it is automatically closed.
     *
     * The closing of a ResultSet object does not close the Blob or Clob objects created
     * by the ResultSet, unless their free method is invoked.
     *
     * Note: A ResultSet object is automatically closed by the Statement object that generated it
     * when that Statement object is closed, re-executed,
     * or is used to retrieve the next result from a sequence of multiple results.
     *
     * @throws SQLException if a database access error occurs
     */
    @Override
    public void close() throws SQLException {
        if (isClosed())
            return;
        String error_msg = MonetNative.monetdbe_result_cleanup(((MonetConnection)this.statement.getConnection()).getDbNative(),nativeResult);
        if (error_msg != null)
            throw new SQLException(error_msg);
        this.closed = true;
        this.columns = null;
        statement.closeIfComplete();
    }

    /**
     * Retrieves the number, types and properties of this ResultSet object's columns.
     *
     * @return the description of this ResultSet object's columns
     */
    @Override
    public ResultSetMetaData getMetaData() throws SQLException {
        checkNotClosed();
        if (metaData == null) {
            metaData = new MonetResultSetMetaData(columns, nColumns);
        }
        return metaData;
    }

    /**
     * Moves the cursor to the given row number in this ResultSet object.
     * If the row number is positive, the cursor moves to the given row number with respect to
     * the beginning of the result set. The first row is row 1, the second is row 2, and so on.
     *
     * If the given row number is negative, the cursor moves to an absolute row position with respect to
     * the end of the result set. For example, calling the method absolute(-1) positions the cursor on the last row;
     * calling the method absolute(-2) moves the cursor to the next-to-last row, and so on.
     *
     * If the row number specified is zero, the cursor is moved to before the first row.
     *
     * An attempt to position the cursor beyond the first/last row in the result set
     * leaves the cursor before the first row or after the last row.
     *
     * @param row the number of the row to which the cursor should move.
     *            A value of zero indicates that the cursor will be positioned before the first row;
     *            a positive number indicates the row number counting from the beginning of the result set;
     *            a negative number indicates the row number counting from the end of the result set
     * @return true if the cursor is moved to a position in this ResultSet object;
     * false if the cursor is before the first row or after the last row
     * @throws SQLException if this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY
     */
    @Override
    public boolean absolute(int row) throws SQLException {
        checkNotClosed();
        if (resultSetType == ResultSet.TYPE_FORWARD_ONLY) {
            throw new SQLException("(Absolute) positioning not allowed on forward " +
                    " only result sets!", "M1M05");
        }
        if (row < 0) {
            row = nRows + row + 1;
        }
        else if (row == 0) {
            curRow = 0;    // before first
            return false;
        }
        else if (row > nRows) {
            curRow = nRows + 1;    // after last
            return false;
        }
        curRow = row;
        return true;
    }

    /**
     * Moves the cursor a relative number of rows, either positive or negative.
     * Attempting to move beyond the first/last row in the result set positions the cursor
     * before/after the the first/last row. Calling relative(0) is valid, but does not change the cursor position.
     *
     * @param rows an int specifying the number of rows to move from the current row;
     *             a positive number moves the cursor forward;
     *             a negative number moves the cursor backward
     * @return true if the cursor is on a row; false otherwise
     * @throws SQLException if this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY
     */
    @Override
    public boolean relative(final int rows) throws SQLException {
        return absolute(curRow + rows);
    }


    /**
     * Moves the cursor to the previous row in this ResultSet object.
     * A ResultSet cursor is initially positioned before the first row;
     * the first call to the method next makes the first row the current row;
     * the second call makes the second row the current row, and so on.
     *
     * When a call to the next method returns false, the cursor is positioned after the last row.
     * Any invocation of a ResultSet method which requires a current row will result in a SQLException being thrown.
     *
     * @return     true if the new current row is valid; false if there are no more rows
     * @throws SQLException if this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY
     */
    @Override
    public boolean next() throws SQLException {
        return relative(1);
    }

    /**
     * Moves the cursor to the previous row in this ResultSet object.
     * When a call to the previous method returns false, the cursor is positioned before the first row.
     * Any invocation of a ResultSet method which requires a current row will result in a SQLException being thrown.
     *
     * @return true if the cursor is now positioned on a valid row; false if the cursor is positioned before the first row
     * @throws SQLException if this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY
     */
    @Override
    public boolean previous() throws SQLException {
        return relative(-1);
    }

    /**
     * Retrieves whether the cursor is before the first row in this ResultSet object.
     *
     * @return true if the cursor is before the first row; false if the cursor is at any other position or the result set contains no rows
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public boolean isBeforeFirst() throws SQLException {
        checkNotClosed();
        return curRow == 0;
    }

    /**
     * Retrieves whether the cursor is after the last row in this ResultSet object.
     *
     * @return true if the cursor is before the first row; false if the cursor is at any other position or the result set contains no rows
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public boolean isAfterLast() throws SQLException {
        checkNotClosed();
        return curRow == nRows + 1;
    }

    /**
     * Retrieves whether the cursor is on the first row of this ResultSet object.
     *
     * @return true if the cursor is on the first row; false otherwise
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public boolean isFirst() throws SQLException {
        checkNotClosed();
        return curRow == 1;
    }

    /**
     * Retrieves whether the cursor is on the last row of this ResultSet object.
     *
     * @return true if the cursor is on the last row; false otherwise
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public boolean isLast() throws SQLException {
        checkNotClosed();
        return curRow == nRows;
    }

    /**
     * Moves the cursor to the front of this ResultSet object, just before the first row.
     *
     * @throws SQLException if this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY
     */
    @Override
    public void beforeFirst() throws SQLException {
        absolute(0);
    }

    /**
     * Moves the cursor to the end of this ResultSet object, just after the last row.
     *
     * @throws SQLException if this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY
     */
    @Override
    public void afterLast() throws SQLException {
        absolute(nRows + 1);
    }

    /**
     * Moves the cursor to the first row in this ResultSet object.
     *
     * @return true if the cursor is on a valid row; false if there are no rows in the result set
     * @throws SQLException if this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY
     */
    @Override
    public boolean first() throws SQLException {
        return absolute(1);
    }

    /**
     * Moves the cursor to the last row in this ResultSet object.
     *
     * @return true if the cursor is on a valid row; false if there are no rows in the result set
     * @throws SQLException if this method is called on a closed result set or the result set type is TYPE_FORWARD_ONLY
     */
    @Override
    public boolean last() throws SQLException {
        return absolute(nRows);
    }

    /**
     * Retrieves the Statement object that produced this ResultSet object.
     * If the result set was generated some other way, such as by a DatabaseMetaData method, this method may return null.
     * @return the Statement object that produced this ResultSet object or null if the result set was produced some other way
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public Statement getStatement() throws SQLException {
        checkNotClosed();
        return statement;
    }

    /**
     * Retrieves the current row number. The first row is number 1, the second number 2, and so on.
     *
     * @return the current row number; 0 if there is no current row
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public int getRow() throws SQLException {
        checkNotClosed();
        return curRow;
    }

    /**
     * Retrieves the type of this ResultSet object. The type is determined by the Statement object that created the result set.
     *
     * @return ResultSet.TYPE_FORWARD_ONLY, ResultSet.TYPE_SCROLL_INSENSITIVE, or ResultSet.TYPE_SCROLL_SENSITIVE
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public int getType() throws SQLException {
        checkNotClosed();
        return resultSetType;
    }

    /**
     * Retrieves the concurrency mode of this ResultSet object. The concurrency used is determined by the Statement object that created the result set.
     *
     * @return the concurrency type, either ResultSet.CONCUR_READ_ONLY or ResultSet.CONCUR_UPDATABLE
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public int getConcurrency() throws SQLException {
        checkNotClosed();
        return concurrency;
    }

    /**
     * Reports whether the last column read had a value of SQL NULL.
     * Note that you must first call one of the getter methods on a column to try to read its value and
     * then call the method wasNull to see if the value read was SQL NULL.
     *
     * @return true if the last column value read was SQL NULL and false otherwise
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public boolean wasNull() throws SQLException {
        checkNotClosed();
        return lastReadWasNull;
    }

    /**
     * Retrieves the holdability of this ResultSet object
     *
     * @return either ResultSet.HOLD_CURSORS_OVER_COMMIT or ResultSet.CLOSE_CURSORS_AT_COMMIT
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public int getHoldability() throws SQLException {
        checkNotClosed();
        return resultSetHoldability;
    }

    /**
     * Gives a hint as to the direction in which the rows in this ResultSet object will be processed.
     * The initial value is determined by the Statement object that produced this ResultSet object.
     * The fetch direction may be changed at any time.
     *
     * @param direction an int specifying the suggested fetch direction;
     *                  one of ResultSet.FETCH_FORWARD, ResultSet.FETCH_REVERSE, or ResultSet.FETCH_UNKNOWN
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public void setFetchDirection(int direction) throws SQLException {
        checkNotClosed();
        this.fetchDirection = direction;
    }

    /**
     * Retrieves the fetch direction for this ResultSet object.
     *
     * @return the current fetch direction for this ResultSet object
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public int getFetchDirection() throws SQLException {
        checkNotClosed();
        return fetchDirection;
    }

    /**
     * Gives the driver a hint as to the number of rows that should be fetched from the database when
     * more rows are needed for this ResultSet object. If the fetch size specified is zero, the driver
     * ignores the value and is free to make its own best guess as to what the fetch size should be.
     * The default value is set by the Statement object that created the result set.
     * The fetch size may be changed at any time.
     *
     * Not used currently.
     *
     * @param rows the number of rows to fetch
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public void setFetchSize(int rows) throws SQLException {
        checkNotClosed();
        this.fetchSize = rows;
    }

    /**
     * Retrieves the fetch size for this ResultSet object.
     *
     * Not used currently.
     *
     * @return the current fetch size for this ResultSet object
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public int getFetchSize() throws SQLException {
        checkNotClosed();
        return fetchSize;
    }

    /**
     * Retrieves the first warning reported by calls on this ResultSet object.
     * Subsequent warnings on this ResultSet object will be chained to the SQLWarning object that this method returns.
     *
     * @return the first SQLWarning object reported or null if there are none
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public SQLWarning getWarnings() throws SQLException {
        checkNotClosed();
        return warnings;
    }

    /**
     * Clears all warnings reported on this ResultSet object.
     * After this method is called, the method getWarnings returns null until a new warning is reported for this ResultSet object.
     *
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public void clearWarnings() throws SQLException {
        checkNotClosed();
        this.warnings = new SQLWarning();
    }

    private final void addWarning(final String reason, final String sqlstate) {
        final SQLWarning warn = new SQLWarning(reason, sqlstate);
        if (warnings == null) {
            warnings = warn;
        } else {
            warnings.setNextWarning(warn);
        }
    }

    /**
     * Retrieves the name of the SQL cursor used by this ResultSet object.
     *
     * @return the SQL name for this ResultSet object's cursor
     * @throws SQLException if this method is called on a closed result set
     */
    @Override
    public String getCursorName() throws SQLException {
        checkNotClosed();
        return name;
    }

    /**
     * Deprecated.
     */
    @Override
    @Deprecated
    public BigDecimal getBigDecimal(int columnIndex, int scale) throws SQLException {
        checkNotClosed();
        if (scale != 0) {
            addWarning("getBigDecimal(int columnIndex, int scale) is deprecated. Please use getBigDecimal(int columnIndex).setScale(int scale)", "");
            return null;
        }
        else {
            return getBigDecimal(columnIndex);
        }
    }

    /**
     * Same as getString(int).
     * @see #getString(int)
     */
    @Override
    public String getNString(int columnIndex) throws SQLException {
        return getString(columnIndex);
    }

    /**
     * Same as getCharacterStream(int).
     * @see #getCharacterStream(int)
     */
    @Override
    public Reader getNCharacterStream(int columnIndex) throws SQLException {
        return getCharacterStream(columnIndex);
    }

    /**
     * Similar to getObject(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getObject(int)
     */
    @Override
    public Object getObject(String columnLabel) throws SQLException {
        return getObject(findColumn(columnLabel));
    }

    /**
     * Similar to getObject(int,Map), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getObject(int, Map)
     */
    @Override
    public Object getObject(String columnLabel, Map<String, Class<?>> map) throws SQLException {
        return getObject(findColumn(columnLabel),map);
    }

    /**
     * Similar to getObject(int,Class), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getObject(int,Class)
     */
    @Override
    public <T> T getObject(String columnLabel, Class<T> type) throws SQLException {
        return getObject(findColumn(columnLabel),type);
    }

    /**
     * Similar to getString(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getString(int)
     */
    @Override
    public String getString(String columnLabel) throws SQLException {
        return getString(findColumn(columnLabel));
    }

    /**
     * Similar to getBoolean(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getBoolean(int)
     */
    @Override
    public boolean getBoolean(String columnLabel) throws SQLException {
        return getBoolean(findColumn(columnLabel));
    }

    /**
     * Similar to getByte(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getByte(int)
     */
    @Override
    public byte getByte(String columnLabel) throws SQLException {
        return getByte(findColumn(columnLabel));
    }

    /**
     * Similar to getShort(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getShort(int)
     */
    @Override
    public short getShort(String columnLabel) throws SQLException {
        return getShort(findColumn(columnLabel));
    }

    /**
     * Similar to getInt(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getInt(int)
     */
    @Override
    public int getInt(String columnLabel) throws SQLException {
        return getInt(findColumn(columnLabel));
    }

    /**
     * Similar to getLong(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getLong(int)
     */
    @Override
    public long getLong(String columnLabel) throws SQLException {
        return getLong(findColumn(columnLabel));
    }

    /**
     * Similar to getFloat(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getFloat(int)
     */
    @Override
    public float getFloat(String columnLabel) throws SQLException {
        return getFloat(findColumn(columnLabel));
    }

    /**
     * Similar to getDouble(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getDouble(int)
     */
    @Override
    public double getDouble(String columnLabel) throws SQLException {
        return getDouble(findColumn(columnLabel));
    }

    /**
     * Deprecated.
     */
    @Deprecated
    @Override
    public BigDecimal getBigDecimal(String columnLabel, int scale) throws SQLException {
        return getBigDecimal(findColumn(columnLabel),scale);
    }

    /**
     * Similar to getBigDecimal(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getBigDecimal(int)
     */
    @Override
    public BigDecimal getBigDecimal(String columnLabel) throws SQLException {
        return getBigDecimal(findColumn(columnLabel));
    }

    /**
     * Similar to getBytes(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getBytes(int)
     */
    @Override
    public byte[] getBytes(String columnLabel) throws SQLException {
        return getBytes(findColumn(columnLabel));
    }

    /**
     * Similar to getDate(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getDate(int)
     */
    @Override
    public Date getDate(String columnLabel) throws SQLException {
        return getDate(findColumn(columnLabel));
    }

    /**
     * Similar to getTime(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getTime(int)
     */
    @Override
    public Time getTime(String columnLabel) throws SQLException {
        return getTime(findColumn(columnLabel));
    }

    /**
     * Similar to getTimestamp(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getTimestamp(int)
     */
    @Override
    public Timestamp getTimestamp(String columnLabel) throws SQLException {
        return getTimestamp(findColumn(columnLabel));
    }

    /**
     * Similar to getBlob(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getBlob(int)
     */
    @Override
    public Blob getBlob(String columnLabel) throws SQLException {
        return getBlob(findColumn(columnLabel));
    }

    /**
     * Similar to getClob(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getClob(int)
     */
    @Override
    public Clob getClob(String columnLabel) throws SQLException {
        return getClob(findColumn(columnLabel));
    }

    /**
     * Similar to getURL(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getURL(int)
     */
    @Override
    public URL getURL(String columnLabel) throws SQLException {
        return getURL(findColumn(columnLabel));
    }

    /**
     * Similar to getNString(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getNString(int)
     */
    @Override
    public String getNString(String columnLabel) throws SQLException {
        return getNString(findColumn(columnLabel));
    }

    /**
     * Similar to getBinaryStream(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getBinaryStream(int)
     */
    @Override
    public InputStream getBinaryStream(String columnLabel) throws SQLException {
        return getBinaryStream(findColumn(columnLabel));
    }

    /**
     * Similar to getCharacterStream(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getCharacterStream(int)
     */
    @Override
    public Reader getCharacterStream(String columnLabel) throws SQLException {
        return getCharacterStream(findColumn(columnLabel));
    }

    /**
     * Similar to getNCharacterStream(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getNCharacterStream(int)
     */
    @Override
    public Reader getNCharacterStream(String columnLabel) throws SQLException {
        return getNCharacterStream(findColumn(columnLabel));
    }

    /**
     * Similar to getDate(int,Calendar), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getDate(int,Calendar)
     */
    @Override
    public Date getDate(String columnLabel, Calendar cal) throws SQLException {
        return getDate(findColumn(columnLabel),cal);
    }

    /**
     * Similar to getTime(int,Calendar), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getTime(int,Calendar)
     */
    @Override
    public Time getTime(String columnLabel, Calendar cal) throws SQLException {
        return getTime(findColumn(columnLabel),cal);
    }

    /**
     * Similar to getTimestamp(int,Calendar), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @see #getTimestamp(int,Calendar)
     */
    @Override
    public Timestamp getTimestamp(String columnLabel, Calendar cal) throws SQLException {
        return getTimestamp(findColumn(columnLabel),cal);
    }

    /**
     * Similar to getBigInteger(int), uses columnLabel instead.
     * @param columnLabel the label for the column specified with the SQL AS clause or the column's name
     * @return the column value; if the value is SQL NULL, the value returned is null
     * @throws SQLException if the columnIndex is not valid or this method is called on a closed result set
     * @see #getBigInteger(int)
     */
    public BigInteger getBigInteger(String columnLabel) throws SQLException {
        return getBigInteger(findColumn(columnLabel));
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public NClob getNClob(int columnIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("getNClob");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public SQLXML getSQLXML(int columnIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("getSQLXML");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public InputStream getAsciiStream(int columnIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("getAsciiStream");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public InputStream getUnicodeStream(int columnIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("getUnicodeStream");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public RowId getRowId(int columnIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("rowId");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public Array getArray(int columnIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("getArray");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public Ref getRef(int columnIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("getRef");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public Ref getRef(String columnLabel) throws SQLException {
        return getRef(findColumn(columnLabel));
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public Array getArray(String columnLabel) throws SQLException {
        return getArray(findColumn(columnLabel));
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public RowId getRowId(String columnLabel) throws SQLException {
        return getRowId(findColumn(columnLabel));
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public NClob getNClob(String columnLabel) throws SQLException {
        return getNClob(findColumn(columnLabel));
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public SQLXML getSQLXML(String columnLabel) throws SQLException {
        return getSQLXML(findColumn(columnLabel));
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public InputStream getAsciiStream(String columnLabel) throws SQLException {
        return getAsciiStream(findColumn(columnLabel));
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public InputStream getUnicodeStream(String columnLabel) throws SQLException {
        return getUnicodeStream(findColumn(columnLabel));
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public boolean rowUpdated() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public boolean rowInserted() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public boolean rowDeleted() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateObject(int columnIndex, Object x, SQLType targetSqlType, int scaleOrLength) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateObject(String columnLabel, Object x, SQLType targetSqlType, int scaleOrLength) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateObject(int columnIndex, Object x, SQLType targetSqlType) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateObject(String columnLabel, Object x, SQLType targetSqlType) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNull(int columnIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBoolean(int columnIndex, boolean x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateByte(int columnIndex, byte x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateShort(int columnIndex, short x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateInt(int columnIndex, int x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateLong(int columnIndex, long x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateFloat(int columnIndex, float x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateDouble(int columnIndex, double x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBigDecimal(int columnIndex, BigDecimal x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateString(int columnIndex, String x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBytes(int columnIndex, byte[] x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateDate(int columnIndex, Date x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateTime(int columnIndex, Time x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateTimestamp(int columnIndex, Timestamp x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateAsciiStream(int columnIndex, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBinaryStream(int columnIndex, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateCharacterStream(int columnIndex, Reader x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateObject(int columnIndex, Object x, int scaleOrLength) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateObject(int columnIndex, Object x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNull(String columnLabel) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBoolean(String columnLabel, boolean x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateByte(String columnLabel, byte x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateShort(String columnLabel, short x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateInt(String columnLabel, int x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateLong(String columnLabel, long x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateFloat(String columnLabel, float x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateDouble(String columnLabel, double x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBigDecimal(String columnLabel, BigDecimal x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateString(String columnLabel, String x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBytes(String columnLabel, byte[] x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateDate(String columnLabel, Date x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateTime(String columnLabel, Time x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateTimestamp(String columnLabel, Timestamp x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateAsciiStream(String columnLabel, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBinaryStream(String columnLabel, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateCharacterStream(String columnLabel, Reader reader, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateObject(String columnLabel, Object x, int scaleOrLength) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateObject(String columnLabel, Object x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void insertRow() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateRow() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void deleteRow() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void refreshRow() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void cancelRowUpdates() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void moveToInsertRow() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void moveToCurrentRow() throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateRef(int columnIndex, Ref x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateRef(String columnLabel, Ref x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBlob(int columnIndex, Blob x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBlob(String columnLabel, Blob x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateClob(int columnIndex, Clob x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateClob(String columnLabel, Clob x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateArray(int columnIndex, Array x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateArray(String columnLabel, Array x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateRowId(int columnIndex, RowId x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateRowId(String columnLabel, RowId x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNString(int columnIndex, String nString) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNString(String columnLabel, String nString) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNClob(int columnIndex, NClob nClob) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNClob(String columnLabel, NClob nClob) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateSQLXML(int columnIndex, SQLXML xmlObject) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateSQLXML(String columnLabel, SQLXML xmlObject) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNCharacterStream(int columnIndex, Reader x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNCharacterStream(String columnLabel, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateAsciiStream(int columnIndex, InputStream x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBinaryStream(int columnIndex, InputStream x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateCharacterStream(int columnIndex, Reader x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateAsciiStream(String columnLabel, InputStream x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBinaryStream(String columnLabel, InputStream x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateCharacterStream(String columnLabel, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBlob(int columnIndex, InputStream inputStream, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    @Override
    public void updateBlob(String columnLabel, InputStream inputStream, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateClob(int columnIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateClob(String columnLabel, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNClob(int columnIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNClob(String columnLabel, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNCharacterStream(int columnIndex, Reader x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNCharacterStream(String columnLabel, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateAsciiStream(int columnIndex, InputStream x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    @Override
    public void updateBinaryStream(int columnIndex, InputStream x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateCharacterStream(int columnIndex, Reader x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateAsciiStream(String columnLabel, InputStream x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBinaryStream(String columnLabel, InputStream x) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateCharacterStream(String columnLabel, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBlob(int columnIndex, InputStream inputStream) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateBlob(String columnLabel, InputStream inputStream) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateClob(int columnIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateClob(String columnLabel, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNClob(int columnIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }

    /**
     * Feature not supported.
     * @throws SQLFeatureNotSupportedException This feature is not supported 
     */
    @Override
    public void updateNClob(String columnLabel, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("update");
    }
}
