package moe.kaede.droids.dispatcher;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;

/**
 * Task dispatcher impl with a single HandlerThread.
 * <p>
 * Use {@link #attach(Handler)} to set an existing handler, which is used to schedule task
 * with the executor or executor an task.
 *
 * @author kaede
 * @version date 16/10/25
 */

public class SimpleDispatcher implements Task.Dispatcher {

    private Handler mHandler;

    public SimpleDispatcher() {
    }

    @Override
    public SimpleDispatcher attach(Handler scheduler) {
        if (mHandler == null) {
            mHandler = scheduler;
        }
        return this;
    }

    @Override
    public void start() {
        if (mHandler == null) {
            synchronized (SimpleDispatcher.this) {
                if (mHandler == null) {
                    HandlerThread thread = new HandlerThread("thread_simple_dispatcher");
                    thread.setPriority(Thread.NORM_PRIORITY);
                    thread.start();
                    mHandler = new Handler(thread.getLooper());
                }
            }
        }
    }

    @Override
    public boolean isRunning() {
        return mHandler != null;
    }

    @Override
    public void post(Runnable runnable) {
        if (mHandler == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        mHandler.post(runnable);
    }

    @Override
    public void post(int what, Runnable runnable) {
        if (mHandler == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        Message message = Message.obtain(mHandler, runnable);
        message.what = what;
        mHandler.sendMessage(message);
    }

    @Override
    public void postDelay(Runnable runnable, long millis) {
        if (mHandler == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (millis < 0) {
            millis = 0;
        }

        mHandler.postDelayed(runnable, millis);
    }

    @Override
    public void postDelay(int what, final Runnable runnable, long millis) {
        if (mHandler == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (millis < 0) {
            millis = 0;
        }

        Message message = Message.obtain(mHandler, runnable);
        message.what = what;
        mHandler.sendMessageDelayed(message, millis);
    }

    @Override
    public boolean has(int what) {
        if (mHandler == null) {
            return false;
        }

        return mHandler.hasMessages(what);
    }

    /**
     * Un support, always return false.
     */
    @Override
    @Deprecated
    public boolean has(Runnable runnable) {
        return false;
    }

    @Override
    public void finish(Runnable runnable) {

    }

    @Override
    public void shutdown() {
        mHandler = null;
    }
}
