/*
 * Copyright (c) 2016. Kaede
 */

package moe.kaede.dispatcher;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Process;
import android.util.Log;

import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Task dispatcher impl with Thread.
 * <p>
 * Use {@link #attach(Handler)} to set an existing handler, which is used to schedule task
 * with the executor.
 *
 * @author kaede
 * @version date 16/10/19
 */

public class ThreadDispatcher implements Task.Dispatcher {

    private final int mDispatcherSize;
    private final AtomicInteger mCount = new AtomicInteger(1);
    private final PriorityBlockingQueue<Runnable> mWorkQueue;

    private Handler mScheduler;
    private DispatcherThread[] mDispatchers;
    private TaskTracker mTaskTracker;

    public ThreadDispatcher(int threadPoolSize) {
        mDispatcherSize = threadPoolSize;
        mWorkQueue = new PriorityBlockingQueue<>();
    }

    public ThreadDispatcher(int threadPoolSize, int capacity) {
        mDispatcherSize = threadPoolSize;
        mWorkQueue = new PriorityBlockingQueue<>(capacity);
    }

    @Override
    public ThreadDispatcher attach(Handler scheduler) {
        if (mScheduler == null) {
            mScheduler = scheduler;
        } else {
            if (BuildConfig.DEBUG) {
                Log.w(TAG, "scheduler has been initialized once.");
            }
        }

        return this;
    }

    @Override
    public void start() {
        if (mDispatchers == null || mDispatchers.length == 0) {
            mDispatchers = new DispatcherThread[mDispatcherSize];

            for (int i = 0; i < mDispatchers.length; i++) {
                mDispatchers[i] = new DispatcherThread();
                mDispatchers[i].start();
            }
        } else {
            if (BuildConfig.DEBUG) {
                Log.w(TAG, "dispatcher has already started once.");
            }
        }
    }

    @Override
    public boolean isRunning() {
        return mDispatchers != null && mDispatchers.length > 0;
    }

    @Override
    public void post(Runnable runnable) {
        if (mDispatchers == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        mWorkQueue.offer(ComparableTask.obtain(this, runnable));
    }

    public void post(int what, Runnable runnable) {
        if (mDispatchers == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (mTaskTracker == null) {
            mTaskTracker = new TaskTracker();
        }

        mTaskTracker.put(what, runnable);
        post(runnable);
    }

    @Override
    public void postDelay(final Runnable runnable, long millis) {
        if (mDispatchers == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (mScheduler == null) {
            if (BuildConfig.DEBUG) {
                Log.d(TAG, "create thread-executor-scheduler");
            }

            HandlerThread thread = new HandlerThread("thread-executor-scheduler");
            thread.setPriority(Thread.NORM_PRIORITY);
            thread.start();
            mScheduler = new Handler(thread.getLooper());
        }

        if (millis < 0) millis = 0;
        mScheduler.postDelayed(new Runnable() {
            @Override
            public void run() {
                if (BuildConfig.DEBUG) {
                    Log.d(TAG, "execute task");
                }
                mWorkQueue.offer(ComparableTask.obtain(ThreadDispatcher.this, runnable));
            }
        }, millis);
    }

    public void postDelay(int what, final Runnable runnable, long millis) {
        if (mDispatchers == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (mTaskTracker == null) {
            mTaskTracker = new TaskTracker();
        }

        mTaskTracker.put(what, runnable);
        postDelay(runnable, millis);
    }

    public boolean has(int what) {
        if (mTaskTracker == null) {
            return false;
        }

        return mTaskTracker.has(what);
    }

    public boolean has(Runnable runnable) {
        if (mTaskTracker == null) {
            return false;
        }

        return mTaskTracker.has(runnable);
    }

    @Override
    public void finish(Runnable runnable) {
        if (mTaskTracker != null && runnable instanceof ComparableTask) {
            ComparableTask wrapper = (ComparableTask) runnable;
            mTaskTracker.remove(wrapper.mRunnable);
        }
    }

    @Override
    public void shutdown() {
        if (mDispatchers != null) {
            for (int i = 0; i < mDispatchers.length; i++) {
                mDispatchers[i].quit();
                mDispatchers[i] = null;
            }

            mDispatchers = null;
        }
    }

    private class DispatcherThread extends Thread {

        public DispatcherThread() {
            String threadName = "ThreadDispatcher #" + mCount.getAndIncrement();
            setName(threadName);

            if (BuildConfig.DEBUG) {
                Log.d(TAG, "create " + threadName);
            }
        }

        @Override
        public void run() {
            Process.setThreadPriority(Process.THREAD_PRIORITY_BACKGROUND);

            while (true) {
                try {
                    Runnable runnable = mWorkQueue.take();
                    runnable.run();

                    if (isInterrupted()) {
                        if (BuildConfig.DEBUG) {
                            Log.d(TAG, "Dispatcher is interrupted.");
                        }
                        break;
                    }


                } catch (InterruptedException e) {
                    if (BuildConfig.DEBUG) {
                        Log.d(TAG, "BlockingQueue is interrupted.");
                    }
                }
            }
        }

        public void quit() {
            interrupt();
        }

    }
}
