/*
 * Copyright (c) 2016. Kaede
 */

package moe.kaede.dispatcher;

import android.os.Handler;

/**
 * Task dispatcher allover api.
 *
 * @author kaede
 * @version date 16/10/26
 */

public class Task {

    /**
     * Task dispatcher interface.
     * Using {@linkplain Dispatchers} to build instance in a simple way.
     *
     * @author kaede
     * @version date 16/10/19
     */
    public interface Dispatcher {
        String TAG = "task.dispatcher";

        /**
         * Attach an existing scheduler to dispatcher, so that we don't need to create an new one.
         */
        Dispatcher attach(Handler scheduler);

        /**
         * Start the dispatcher before it can work.
         */
        void start();

        /**
         * Whether or not the dispatcher is working.
         */
        boolean isRunning();

        /**
         * Add an task to the dispatcher.
         * Note that you should call {@linkplain #start()} before this.
         */
        void post(Runnable runnable);

        /**
         * Add an task to the dispatcher with id.
         */
        void post(int what, Runnable runnable);

        /**
         * Schedule an task to the dispatcher.
         * Note that you should call {@linkplain #start()} before this.
         */
        void postDelay(Runnable runnable, long millis);

        /**
         * Schedule an task to the dispatcher with id.
         */
        void postDelay(int what, Runnable runnable, long millis);

        /**
         * Check if there exists the task with the specific id.
         */
        boolean has(int what);

        /**
         * Check if there exists the task with the specific runnable.
         */
        boolean has(Runnable runnable);

        /**
         * When a task is finished, it will call this method.
         * Note that you should not call this method directly.
         */
        void finish(Runnable runnable);

        /**
         * Terminate the dispatcher.
         */
        void shutdown();
    }

    /**
     * Build utility for {@linkplain Dispatcher}
     */
    public static class Dispatchers {
        public static SimpleDispatcher newSimpleDispatcher() {
            return new SimpleDispatcher();
        }

        public static ExecutorDispatcher newExecutorDispatcher(int threadPoolSize) {
            return new ExecutorDispatcher(threadPoolSize);
        }

        public static ExecutorDispatcher newExecutorDispatcher(int threadPoolSize, int capacity) {
            return new ExecutorDispatcher(threadPoolSize, capacity);
        }

        public static ThreadDispatcher newThreadDispatcher(int threadPoolSize) {
            return new ThreadDispatcher(threadPoolSize);
        }

        public static ThreadDispatcher newThreadDispatcher(int threadPoolSize, int capacity) {
            return new ThreadDispatcher(threadPoolSize, capacity);
        }
    }
}
