/*
 * Copyright (c) 2016. Kaede
 */

package moe.kaede.dispatcher;

import android.os.Build;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.util.Log;

/**
 * Task dispatcher impl with a single HandlerThread.
 * <p>
 * Use {@link #attach(Handler)} to set an existing handler, which is used to schedule task
 * with the executor or executor an task.
 *
 * @author kaede
 * @version date 16/10/25
 */

public class SimpleDispatcher implements Task.Dispatcher {

    private Handler mHandler;
    private boolean mUsingBuiltinHandler;

    public SimpleDispatcher() {
    }

    @Override
    public SimpleDispatcher attach(Handler scheduler) {
        if (mHandler == null) {
            mHandler = scheduler;
            mUsingBuiltinHandler = false;
        } else {
            if (BuildConfig.DEBUG) {
                Log.w(TAG, "scheduler has been initialized once.");
            }
        }

        return this;
    }

    @Override
    public void start() {
        if (mHandler == null) {
            synchronized (SimpleDispatcher.this) {
                if (mHandler == null) {
                    HandlerThread thread = new HandlerThread("thread_simple_dispatcher");
                    thread.setPriority(Thread.NORM_PRIORITY);
                    thread.start();
                    mHandler = new Handler(thread.getLooper());
                    mUsingBuiltinHandler = true;
                }
            }
        }
    }

    @Override
    public boolean isRunning() {
        return mHandler != null;
    }

    @Override
    public void post(Runnable runnable) {
        if (mHandler == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        mHandler.post(runnable);
    }

    @Override
    public void post(int what, Runnable runnable) {
        if (mHandler == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        Message message = Message.obtain(mHandler, runnable);
        message.what = what;
        mHandler.sendMessage(message);
    }

    @Override
    public void postDelay(Runnable runnable, long millis) {
        if (mHandler == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (millis < 0) {
            millis = 0;
        }

        mHandler.postDelayed(runnable, millis);
    }

    @Override
    public void postDelay(int what, final Runnable runnable, long millis) {
        if (mHandler == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (millis < 0) {
            millis = 0;
        }

        Message message = Message.obtain(mHandler, runnable);
        message.what = what;
        mHandler.sendMessageDelayed(message, millis);
    }

    @Override
    public boolean has(int what) {
        if (mHandler == null) {
            return false;
        }

        return mHandler.hasMessages(what);
    }

    /**
     * Un support, always return false.
     */
    @Override
    @Deprecated
    public boolean has(Runnable runnable) {
        return false;
    }

    @Override
    public void finish(Runnable runnable) {

    }

    @Override
    public void shutdown() {
        if (mHandler != null) {
            if (mUsingBuiltinHandler) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2) {
                    mHandler.getLooper().quitSafely();
                } else {
                    mHandler.getLooper().quit();
                }
            }
            mHandler = null;
        }
    }
}
