/*
 * Copyright (c) 2016. Kaede
 */

package moe.kaede.dispatcher;

import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;


/**
 * Task dispatcher impl with ExecutorService.
 * <p>
 * Worker dispatcher with an {@link ExecutorService}. Note that if the executor's work
 * queue is bounded, the excessive task will be added to the pending waiting for a working
 * task to be finished.
 * <p>
 * Use {@link #attach(ExecutorService)} to set an existing executor, but if you do not use embed
 * executor, you have to deal with the bounded issue.
 * <p>
 * Use {@link #attach(Handler)} to set an existing handler, which is used to schedule task
 * with the executor.
 *
 * @author Kaede
 * @version 2016-10-18
 */
public class ExecutorDispatcher implements Task.Dispatcher, ThreadFactory, RejectedExecutionHandler {

    private final int mCorePoolSize;
    private final int mMaximumPoolSize;
    private final long mKeepAliveTime;

    private final AtomicInteger mCount = new AtomicInteger(1);
    private final PriorityBlockingQueue<Runnable> mWorkQueue;

    private Handler mScheduler;
    private ExecutorService mExecutor;
    private TaskTracker mTaskTracker;

    public ExecutorDispatcher(int threadPoolSize) {
        mCorePoolSize = threadPoolSize;
        mMaximumPoolSize = threadPoolSize * 2;
        // time to keep thread when it's idle. 30s
        mKeepAliveTime = 30L;
        mWorkQueue = new PriorityBlockingQueue<>();
    }

    public ExecutorDispatcher(int threadPoolSize, int capacity) {
        mCorePoolSize = threadPoolSize;
        mMaximumPoolSize = threadPoolSize * 2;
        mKeepAliveTime = 30L;
        mWorkQueue = new PriorityBlockingQueue<>(capacity);
    }

    @Override
    public ExecutorDispatcher attach(Handler scheduler) {
        if (mScheduler == null) {
            mScheduler = scheduler;
        } else {
            if (BuildConfig.DEBUG) {
                Log.w(TAG, "scheduler has been initialized once.");
            }
        }

        return this;
    }

    /**
     * Attach an existing {@link ExecutorService} to dispatcher, so that we don't need to create
     * an new one.
     */
    public ExecutorDispatcher attach(ExecutorService executor) {
        if (mExecutor == null) {
            mExecutor = executor;
        } else {
            if (BuildConfig.DEBUG) {
                Log.w(TAG, "executor has been initialized once.");
            }
        }

        return this;
    }

    public ExecutorService getExecutor() {
        return mExecutor;
    }

    @Override
    public void start() {
        if (mExecutor == null) {
            mExecutor = new ThreadPoolExecutor(mCorePoolSize, mMaximumPoolSize, mKeepAliveTime,
                    TimeUnit.SECONDS, mWorkQueue, this, this);

            ((ThreadPoolExecutor) mExecutor).allowCoreThreadTimeOut(true);

        } else {
            if (BuildConfig.DEBUG) {
                Log.w(TAG, "dispatcher has already started once.");
            }
        }
    }

    @Override
    public boolean isRunning() {
        return mExecutor != null && !mExecutor.isShutdown();
    }

    @Override
    public void post(Runnable runnable) {
        if (mExecutor == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        mExecutor.execute(ComparableTask.obtain(this, runnable));
    }

    public void post(int what, Runnable runnable) {
        if (mExecutor == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (mTaskTracker == null) {
            mTaskTracker = new TaskTracker();
        }

        mTaskTracker.put(what, runnable);
        post(runnable);
    }

    @Override
    public void postDelay(final Runnable runnable, long millis) {
        if (mExecutor == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (mScheduler == null) {
            if (BuildConfig.DEBUG) {
                Log.d(TAG, "create thread-executor-scheduler");
            }

            HandlerThread thread = new HandlerThread("thread-executor-scheduler");
            thread.setPriority(Thread.NORM_PRIORITY);
            thread.start();
            mScheduler = new Handler(thread.getLooper());
        }

        if (millis < 0) millis = 0;
        mScheduler.postDelayed(new Runnable() {
            @Override
            public void run() {
                if (BuildConfig.DEBUG) Log.d(TAG, "execute task");
                mExecutor.execute(ComparableTask.obtain(ExecutorDispatcher.this, runnable));
            }
        }, millis);
    }

    public void postDelay(int what, final Runnable runnable, long millis) {
        if (mExecutor == null) {
            throw new IllegalStateException("pls call #start to initialize.");
        }

        if (mTaskTracker == null) {
            mTaskTracker = new TaskTracker();
        }

        mTaskTracker.put(what, runnable);
        postDelay(runnable, millis);
    }

    public boolean has(int what) {
        if (mTaskTracker == null) {
            return false;
        }

        return mTaskTracker.has(what);
    }

    public boolean has(Runnable runnable) {
        if (mTaskTracker == null) {
            return false;
        }

        return mTaskTracker.has(runnable);
    }

    @Override
    public void finish(Runnable runnable) {
        if (BuildConfig.DEBUG) {
            Log.d(TAG, "task finish");
        }

        if (mTaskTracker != null && runnable instanceof ComparableTask) {
            ComparableTask wrapper = (ComparableTask) runnable;
            mTaskTracker.remove(wrapper.mRunnable);
        }

    }

    @Override
    public void shutdown() {
        if (mExecutor != null) {
            mExecutor.shutdown();
        }
    }

    @Override
    public Thread newThread(Runnable r) {
        Thread thread = new Thread(r, "ExecutorDispatcher #" + mCount.getAndIncrement());
        thread.setPriority(Thread.NORM_PRIORITY);

        if (BuildConfig.DEBUG) {
            Log.d(TAG, "executor new thread : " + thread.getName());
        }
        return thread;
    }

    @Override
    public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
        // Offer pending queue when the executor's working queue is bounded.
        if (BuildConfig.DEBUG) {
            Log.w(TAG, "bounded work queue, discard oldest task.");
        }

        /**
         * Using {@link java.util.concurrent.ThreadPoolExecutor.DiscardOldestPolicy}.
         */
        if (!executor.isShutdown()) {
            executor.getQueue().poll();
            executor.execute(r);
        }
    }

}
