
package com.xiaomi.market.sdk;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.DialogInterface.OnClickListener;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.content.pm.PackageManager.NameNotFoundException;
import android.net.Uri;
import android.os.AsyncTask;
import android.text.TextUtils;

import com.xiaomi.market.sdk.Connection.NetworkError;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.util.List;

import static com.xiaomi.market.sdk.UpdateStatus.STATUS_FAILED;
import static com.xiaomi.market.sdk.UpdateStatus.STATUS_LOCAL_APP_FAILED;
import static com.xiaomi.market.sdk.UpdateStatus.STATUS_NO_NET;
import static com.xiaomi.market.sdk.UpdateStatus.STATUS_NO_UPDATE;
import static com.xiaomi.market.sdk.UpdateStatus.STATUS_NO_WIFI;
import static com.xiaomi.market.sdk.UpdateStatus.STATUS_UPDATE;

public class XiaomiUpdateAgent {
    private static boolean mIsInited = false;
    private static boolean mIsLoading = false;
    private static final String TAG = "MarketUpdateAgent";

    private static WeakReference<Context> mContext = new WeakReference<Context>(null);
    private static boolean mAutoPopup = true;
    private static boolean mCheckUpdateOnlyWifi = false;
    private static boolean mIsPathcerLibraryLoaded = false;

    private static LocalAppInfo mAppInfo;
    private static UpdateInfo mUpdateInfo;
    private static XiaomiUpdateListener mUpdateListener;
    private static Constants.UpdateMethod mUpdateMethod = Utils.isMiuiPad() ? Constants.UpdateMethod.DOWNLOAD_MANAGER : Constants.UpdateMethod.MARKET;
    private static boolean sDebug;

    public synchronized static void update(Context context) {
        boolean debug = false;
        try {
            Class<?> buildConfigClazz = Class.forName(context.getPackageName() + ".BuildConfig");
            Field fDebug = buildConfigClazz.getDeclaredField("DEBUG");
            debug = fDebug.getBoolean(buildConfigClazz);
        } catch (Exception e) {
            e.printStackTrace();
        }
        update(context, debug);
    }

    public synchronized static void update(Context context, boolean debug) {
        if (context == null || mIsLoading) {
            return;
        }
        sDebug = debug;
        mIsLoading = true;
        Client.init(context);
        AppGlobal.setContext(context);
        mContext = new WeakReference<Context>(context);

        if (!mIsInited) {
            mAppInfo = null;
            mUpdateInfo = null;
            Constants.configURL();
            mIsInited = true;
        }
        new CheckUpdateTask().execute();
    }

    public static void setUseInternationalHost(boolean useInternational) {
        Constants.setUseInternalProductUrl(useInternational);
        Constants.configURL();
    }

    public static void arrange() {
        Context context = mContext.get();
        if (context == null) {
            return;
        }
        Client.init(context);
        openMarketOrArrange();
    }

    public static void setUpdateAutoPopup(boolean autoPopup) {
        mAutoPopup = autoPopup;
    }

    public static void setCheckUpdateOnlyWifi(boolean checkUpdateOnlyWifi) {
        mCheckUpdateOnlyWifi = checkUpdateOnlyWifi;
    }

    public static void setUpdateListener(XiaomiUpdateListener updateListener) {
        mUpdateListener = updateListener;
    }

    public static int getSDKVersion(){
        return Client.XIAOMI_SDK_VERSION_CODE;
    }

    /**
     * set priority updateMethod , con`t ensure use mi market to update apps
     *
     * @param method
     */
    public static void setUpdateMethod(Constants.UpdateMethod method) {
        mUpdateMethod = method;
    }

    static Context getContext() {
        return mContext.get();
    }

    static LocalAppInfo getAppInfo(Context context) {
        LocalAppInfo appInfo = LocalAppInfo.get(context.getPackageName());
        PackageManager packageManager = context.getPackageManager();
        PackageInfo pkgInfo = null;
        try {
            pkgInfo = packageManager.getPackageInfo(appInfo.packageName,
                    PackageManager.GET_SIGNATURES);
        } catch (NameNotFoundException e) {
            Log.e(TAG, "get package info failed");
        }
        if (pkgInfo != null && pkgInfo.applicationInfo != null) {
            appInfo.displayName = packageManager.getApplicationLabel(pkgInfo.applicationInfo)
                    .toString();
            appInfo.versionCode = pkgInfo.versionCode;
            appInfo.versionName = pkgInfo.versionName;
            appInfo.signature = Coder
                    .encodeMD5(String.valueOf(pkgInfo.signatures[0].toChars()));
            appInfo.sourceDir = pkgInfo.applicationInfo.sourceDir;
            appInfo.sourceMD5 = Coder.encodeMD5(new File(appInfo.sourceDir));
            return appInfo;
        }
        return null;
    }

    static class UpdateInfo {
        String host;
        int source;
        int fitness;
        String updateLog;
        int versionCode;
        String versionName;
        String apkUrl;
        String apkHash;
        long apkSize;
        String diffUrl = "";
        String diffHash = "";
        long diffSize;

        @Override
        public String toString() {
            return "UpdateInfo:\nhost = " + host + "\nfitness = " + fitness + "\nupdateLog = "
                    + updateLog + "\nversionCode = " + versionCode + "\nversionName = "
                    + versionName + "\napkUrl = " + apkUrl + "\napkHash = " + apkHash
                    + "\napkSize = " + apkSize + "\ndiffUrl = " + diffUrl + "\ndiffHash = "
                    + diffHash + "\ndiffSize = " + diffSize;
        }
    }

    private static class CheckUpdateTask extends AsyncTask<Void, Void, Integer> {
        @Override
        protected void onPreExecute() {
            Log.d(TAG, "start to check update");
            if (!mIsPathcerLibraryLoaded) {
                mIsPathcerLibraryLoaded = Patcher.tryLoadLibrary();
            }
        }

        @Override
        protected Integer doInBackground(Void... params) {
            Context context = mContext.get();
            if (context == null) {
                return STATUS_FAILED;
            }
            if (!Utils.isConnected(context)) {
                return STATUS_NO_NET;
            } else if (!Utils.isWifiConnected(context) && mCheckUpdateOnlyWifi) {
                return STATUS_NO_WIFI;
            }

            mAppInfo = getAppInfo(context);
            if (mAppInfo == null) {
                return STATUS_LOCAL_APP_FAILED;
            }

            Connection conn = new Connection(Constants.UPDATE_URL);
            Connection.Parameter parameters = conn.new Parameter();
            parameters.add(Constants.JSON_FILTER_INFO, getFilterParams());
            parameters.add(Constants.JSON_PACKAGE_NAME, mAppInfo.packageName);
            parameters.add(Constants.JSON_VERSION_CODE, mAppInfo.versionCode + "");
            parameters.add(Constants.JSON_OLD_APK_HASH, mAppInfo.sourceMD5);
            parameters.add(Constants.JSON_SIGNATURE, mAppInfo.signature);
            parameters.add(Constants.JSON_SDK_VERSION, String.valueOf(Client.SDK_VERSION));
            parameters.add(Constants.JSON_VERSION, Client.SYSTEM_VERSION);
            parameters.add(Constants.JSON_LANGUAGE, Client.LANGUAGE);
            parameters.add(Constants.JSON_COUNTRY, Client.COUNTRY);
            parameters.add(Constants.JSON_ANDROID_ID, Client.ANDROID_ID);
            parameters.add(Constants.JSON_XIAOMI_SDK_VERSION, Client.XIAOMI_SDK_VERSION_CODE + "");
            parameters.add(Constants.JSON_DEBUG, sDebug ? "1" : "0");

            JSONObject obj = null;
            if (NetworkError.OK == conn.requestJSON()) {
                obj = conn.getResponse();
                mUpdateInfo = parseUpdateInfo(obj);
                if (mUpdateInfo != null) {
                    Log.i(TAG, mUpdateInfo.toString());
                    return mUpdateInfo.fitness == 0 ? STATUS_UPDATE : STATUS_NO_UPDATE;
                }
            }
            return STATUS_FAILED;

        }

        @Override
        protected void onPostExecute(Integer status) {
            mIsLoading = false;
            if (!mAutoPopup) {
                UpdateResponse response = new UpdateResponse();
                if (status == STATUS_UPDATE) {
                    response.updateLog = mUpdateInfo.updateLog;
                    response.versionCode = mUpdateInfo.versionCode;
                    response.versionName = mUpdateInfo.versionName;
                    response.apkSize = mUpdateInfo.apkSize;
                    response.apkHash = mUpdateInfo.apkHash;
                    response.diffSize = mUpdateInfo.diffSize;
                    response.path = Connection.connect(mUpdateInfo.host, mUpdateInfo.apkUrl);
                }
                if (mUpdateListener != null) {
                    mUpdateListener.onUpdateReturned(status, response);
                }
            } else {
                switch (status) {
                    case STATUS_UPDATE:
                        new CheckDownloadTask().execute();
                        break;
                    default:
                        break;
                }
            }
        }

        private String getFilterParams() {
            JSONObject obj = new JSONObject();
            try {
                obj.put(Constants.JSON_SCREEN_SIZE, Client.DISPLAY_WIDTH + "*"
                        + Client.DISPLAY_HEIGHT);
                obj.put(Constants.JSON_RESOLUTION, Client.DISPLAY_RESOLUTION);
                obj.put(Constants.JSON_DENSITY, Client.DISPLAY_DENSITY);
                obj.put(Constants.JSON_TOUCH_SCREEN, Client.TOUCH_SCREEN);
                obj.put(Constants.JSON_GLES_VERSION, Client.GLES_VERSION);
                obj.put(Constants.JSON_FEATURE, Client.FEATURE);
                obj.put(Constants.JSON_LIBRARY, Client.LIBRARY);
                obj.put(Constants.JSON_GL_EXTENSION, Client.GL_EXTENSION);
                obj.put(Constants.JSON_SDK, Client.SDK_VERSION);
                obj.put(Constants.JSON_SYSTEM_VERSION, Client.SYSTEM_VERSION);
                obj.put(Constants.JSON_SYSTEM_RELEASE, Client.RELEASE);
            } catch (JSONException e) {
                return "";
            }
            return obj.toString();
        }

        private UpdateInfo parseUpdateInfo(JSONObject obj) {
            if (obj == null) {
                Log.e(TAG, "update info json obj null");
                return null;
            }

            if (Utils.DEBUG) {
                Log.d(TAG, "updateInfo : " + obj.toString());
            }

            UpdateInfo updateInfo = new UpdateInfo();
            try {
                updateInfo.host = obj.getString(Constants.HOST);
                updateInfo.fitness = obj.getInt(Constants.FITNESS);
                updateInfo.source = obj.getInt(Constants.SOURCE);
                updateInfo.updateLog = obj.getString(Constants.UPDATE_LOG);
                updateInfo.versionCode = obj.getInt(Constants.VERSION_CODE);
                updateInfo.versionName = obj.getString(Constants.VERSION_NAME);
                updateInfo.apkUrl = obj.getString(Constants.APK_URL);
                updateInfo.apkHash = obj.getString(Constants.APK_HASH);
                updateInfo.apkSize = obj.getLong(Constants.APK_SIZE);

                // 若Patcher库没被加载进来,则直接不读取增量更新信息
                if (mIsPathcerLibraryLoaded) {
                    updateInfo.diffUrl = obj.getString(Constants.DIFF_URL);
                    updateInfo.diffHash = obj.getString(Constants.DIFF_HASH);
                    updateInfo.diffSize = obj.getLong(Constants.DIFF_SIZE);
                }
            } catch (JSONException e) {
                Log.e(TAG, "get update info failed : " + e.toString());
                Log.e(TAG, "original content : " + obj.toString());
                return null;
            }

            return updateInfo;
        }

        private static class CheckDownloadTask extends AsyncTask<Void, Void, Boolean> {

            @Override
            protected Boolean doInBackground(Void... params) {
                Context context = mContext.get();
                if (context == null) {
                    return false;
                }
                return DownloadInstallManager.getManager(context).isDownloading(mAppInfo);
            }

            @Override
            protected void onPostExecute(Boolean result) {
                if (!result) {
                    showUpdateDialog();
                }
            }
        }

        private static void showUpdateDialog() {
            Context context = mContext.get();
            if (context == null) {
                return;
            }
            if (context instanceof Activity && ((Activity) context).isFinishing()) {
                Log.e(TAG, "activity not running!");
                return;
            }
            new AlertDialog.Builder(context)
                    .setTitle(context.getString(R.string.xiaomi_update_sdk_dialog_title, mAppInfo.displayName))
                    .setMessage(
                            TextUtils.isEmpty(mUpdateInfo.diffUrl) ?
                                    context.getString(R.string.xiaomi_update_sdk_dialog_message,
                                            mUpdateInfo.versionName,
                                            Utils.getByteString(mUpdateInfo.apkSize, context)) :
                                    context.getString(R.string.xiaomi_update_sdk_dialog_message_diff,
                                            mUpdateInfo.versionName,
                                            Utils.getByteString(mUpdateInfo.diffSize, context)))
                    .setNegativeButton(android.R.string.cancel, null)
                    .setPositiveButton(android.R.string.ok, new OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            openMarketOrArrange();
                        }
                    }).show();
        }
    }

    static void openMarketOrArrange() {
        Context context = mContext.get();
        if (context == null || mUpdateInfo == null || mAppInfo == null) {
            return;
        }
        boolean useMarket = mUpdateMethod.equals(Constants.UpdateMethod.MARKET);
        if (useMarket && mUpdateInfo.source != 1 && Utils.isMiuiMarketExisted(context)) {
            Intent intent = new Intent(Intent.ACTION_VIEW,
                    Uri.parse("market://details?back=true&id=" + mAppInfo.packageName));
            intent.setClassName("com.xiaomi.market", "com.xiaomi.market.ui.AppDetailActivity");
            List<ResolveInfo> activityList = context.getPackageManager()
                    .queryIntentActivities(intent, 0);
            if (activityList != null && activityList.size() == 1) {
                ResolveInfo activityInfo = activityList.get(0);
                if (activityInfo.activityInfo != null && activityInfo.activityInfo.exported
                        && activityInfo.activityInfo.enabled) {
                    context.startActivity(intent);
                    return;
                }
            }
        }
        DownloadInstallManager.getManager(context).arrange(mAppInfo, mUpdateInfo);
    }

}
