
package com.xiaomi.market.sdk;

import android.annotation.SuppressLint;
import android.app.DownloadManager;
import android.app.DownloadManager.Query;
import android.app.DownloadManager.Request;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.net.Uri;
import android.os.Environment;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.support.v4.content.FileProvider;
import android.text.TextUtils;

import com.xiaomi.market.sdk.Constants.Update;
import com.xiaomi.market.sdk.XiaomiUpdateAgent.UpdateInfo;
import static com.xiaomi.market.sdk.DownloadInstallResultNotifier.*;

import java.io.File;
import java.lang.reflect.Method;

public class DownloadInstallManager {
    private static final String TAG = "MarketUpdateDownload";
    // column file path in 2.3 DownloadManager
    private static final String COLUMN_FILE_PATH = "file_path";

    public static DownloadInstallManager sDownloadInstallManager;

    private UpdateInfo mUpdateInfo;
    private LocalAppInfo mAppInfo;
    private long mDownloadId = -1;

    private static DownloadManager mDownloadManager;
    private HandlerThread mWorkerThread;
    private WorkerHandler mWorkerHandler;
    private Context mContext;

    private DownloadInstallManager(Context context) {
        Client.init(context);
        mContext = context.getApplicationContext();
        initDownloadManager();

        mWorkerThread = new HandlerThread("Worker Thread");
        mWorkerThread.start();
        mWorkerHandler = new WorkerHandler(mWorkerThread.getLooper());
    }

    private void initDownloadManager() {
        mDownloadManager = (DownloadManager) mContext
                .getSystemService(Context.DOWNLOAD_SERVICE);
        if (Client.isLaterThanN()) {
            try {
                Method method = mDownloadManager.getClass().getDeclaredMethod("setAccessFilename", boolean.class);
                method.setAccessible(true);
                method.invoke(mDownloadManager, true);
            } catch (Exception e) {
                Log.e(TAG, e.getMessage(), e);
            }
        }
    }

    public void arrange(LocalAppInfo appInfo, UpdateInfo updateInfo) {
        if (updateInfo == null || appInfo == null) {
            return;
        }
        mUpdateInfo = updateInfo;
        mAppInfo = appInfo;
        mWorkerHandler.arrange();
    }

    public void handleDownloadComplete(long downloadId) {
        if (downloadId < 0 || (mDownloadId != downloadId)) {
            return;
        }
        DownloadManagerInfo downloadManagerInfo = DownloadManagerInfo.find(mDownloadId);
        if (downloadManagerInfo == null
                || downloadManagerInfo.status == DownloadManager.STATUS_FAILED
                || TextUtils.isEmpty(downloadManagerInfo.downloadFilePath)) {
            return;
        }
        mWorkerHandler.install(downloadManagerInfo.downloadFilePath, !TextUtils.isEmpty(mUpdateInfo.diffUrl));
    }

    public static synchronized DownloadInstallManager getManager(Context context) {
        if (sDownloadInstallManager == null) {
            sDownloadInstallManager = new DownloadInstallManager(context);
        }
        return sDownloadInstallManager;
    }

    public boolean isDownloading(LocalAppInfo appInfo) {
        Cursor cursor = SDKDatabaseHelper.getHelper(mContext).query(Update.TABLE,
                Update.UPDATE_PROJECTION,
                Update.PACKAGE_NAME + "=?", new String[]{
                        appInfo.packageName
                }, null, null, null);
        long downloadId = -1;
        try {
            if (cursor != null && cursor.moveToFirst()) {
                downloadId = cursor.getLong(cursor.getColumnIndex(Update.DOWNLOAD_ID));
            }
            if (downloadId == -1) {
                return false;
            }
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }

        Query query = new Query();
        query.setFilterById(downloadId);
        Cursor downloadCursor = mDownloadManager.query(query);
        int status = -1;
        try {
            if (downloadCursor != null && downloadCursor.moveToFirst()) {
                status = downloadCursor.getInt(downloadCursor
                        .getColumnIndexOrThrow(DownloadManager.COLUMN_STATUS));
            }
            if (status != DownloadManager.STATUS_PAUSED && status != DownloadManager.STATUS_PENDING
                    && status != DownloadManager.STATUS_RUNNING) {
                return false;
            }
        } finally {
            if (downloadCursor != null) {
                downloadCursor.close();
            }
        }
        return true;
    }

    public class WorkerHandler extends Handler {
        public WorkerHandler(Looper looper) {
            super(looper);
        }

        public void reloadDownloadTasks() {
            if (mAppInfo == null || mUpdateInfo == null) {
                mAppInfo = XiaomiUpdateAgent.getAppInfo(mContext);
                if (mAppInfo == null) {
                    return;
                }
                Cursor cursor = null;
                try {
                    cursor = SDKDatabaseHelper.getHelper(mContext).query(Update.TABLE,
                            Update.UPDATE_PROJECTION, Update.PACKAGE_NAME + "=?", new String[]{
                                    mAppInfo.packageName
                            }, null, null, null);
                    if (cursor != null && cursor.moveToFirst()) {
                        mDownloadId = cursor.getLong(cursor.getColumnIndex(Update.DOWNLOAD_ID));
                        UpdateInfo restoreUpdateInfo = new UpdateInfo();
                        restoreUpdateInfo.versionCode = cursor.getInt(cursor
                                .getColumnIndex(Update.VERSION_CODE));
                        restoreUpdateInfo.apkUrl = cursor.getString(cursor
                                .getColumnIndex(Update.APK_URL));
                        restoreUpdateInfo.apkHash = cursor.getString(cursor
                                .getColumnIndex(Update.APK_HASH));
                        restoreUpdateInfo.diffUrl = cursor.getString(cursor
                                .getColumnIndex(Update.DIFF_URL));
                        restoreUpdateInfo.diffHash = cursor.getString(cursor
                                .getColumnIndex(Update.DIFF_HASH));
                        mUpdateInfo = restoreUpdateInfo;
                    } else {
                        return;
                    }
                } finally {
                    if (cursor != null) {
                        cursor.close();
                    }
                }
            }
        }

        public void arrange() {
            post(new Runnable() {
                @Override
                public void run() {
                    if (!Utils.hasExternalStorage(true)) {
                        DownloadInstallResultNotifier.notifyResult(ERROR_SDCARD_NOT_AVAILABLE);
                        return;
                    }

                    if (mDownloadManager == null) {
                        return;
                    }

                    File apkSaveFile = generateApkSaveFile();
                    if (apkSaveFile == null) {
                        return;
                    }

                    if (apkSaveFile.exists()) {
                        if (TextUtils.equals(Coder.encodeMD5(apkSaveFile), mUpdateInfo.apkHash)) {
                            install(apkSaveFile.getAbsolutePath(), false);
                            return;
                        }
                        apkSaveFile.delete();
                    }

                    downloadApk(getDownloadUri(), apkSaveFile.getAbsolutePath());
                }
            });
        }

        private void downloadApk(Uri downloadUri, String savedPath) {
            Uri apkUri = Uri.parse("file://" + savedPath);
            Request request = new Request(downloadUri);
            request.setMimeType("application/apk-ota");
            request.setTitle(mAppInfo.displayName);
            if (apkUri != null) {
                request.setDestinationUri(apkUri);
            }

            try {
                mDownloadId = mDownloadManager.enqueue(request);
            } catch (Throwable th) {
                Log.e(TAG, th.toString());
                DownloadInstallResultNotifier.notifyResult(ERROR_CONNECT_DOWNLOAD_MANAGER);
                return;
            }

            // store the download info into the database
            ContentValues contentValues = new ContentValues();
            contentValues.put(Update.PACKAGE_NAME, mAppInfo.packageName);
            contentValues.put(Update.DOWNLOAD_ID, mDownloadId);
            contentValues.put(Update.VERSION_CODE, mUpdateInfo.versionCode);
            contentValues.put(Update.APK_URL, mUpdateInfo.apkUrl);
            contentValues.put(Update.APK_HASH, mUpdateInfo.apkHash);
            contentValues.put(Update.DIFF_URL, mUpdateInfo.diffUrl);
            contentValues.put(Update.DIFF_HASH, mUpdateInfo.diffHash);
            contentValues.put(Update.APK_PATH, savedPath);
            SDKDatabaseHelper.getHelper(AppGlobal.getContext()).insertOrUpdateUpdate(contentValues);
        }

        private Uri getDownloadUri() {
            String downloadUrl;
            if (TextUtils.isEmpty(mUpdateInfo.diffUrl)) {
                downloadUrl = Connection.connect(mUpdateInfo.host, mUpdateInfo.apkUrl);
            } else {
                downloadUrl = Connection.connect(mUpdateInfo.host, mUpdateInfo.diffUrl);
            }
            return Uri.parse(downloadUrl);
        }

        private File generateApkSaveFile() {
            File environmentDownloadDir = mContext.getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS);
            if (environmentDownloadDir == null) {
                return null;
            }
            File externalApkDir = new File(environmentDownloadDir.getAbsolutePath()
                    + "/xiaomi_update_sdk");
            if (!externalApkDir.exists()) {
                externalApkDir.mkdirs();
            }
            File file = new File(externalApkDir.getAbsolutePath() + "/"
                    + mAppInfo.packageName + "_" + mUpdateInfo.versionCode + ".apk");
            return file;
        }

        public void install(final String originApkPath, final boolean diffUpdate) {
            post(new Runnable() {
                @Override
                public void run() {
                    if (TextUtils.isEmpty(originApkPath)) {
                        return;
                    }

                    reloadDownloadTasks();

                    String apkFilePath = originApkPath;
                    if (diffUpdate) {
                        apkFilePath = getPatchedApk(apkFilePath, mUpdateInfo.diffHash);
                    }

                    if (!verify(apkFilePath)) {
                        Log.e(TAG, "verify downloaded apk failed");
                        return;
                    }

                    launchPackageInstaller(apkFilePath);
                }
            });
        }

        private void launchPackageInstaller(String apkPath) {
            Uri installUri = generateInstallUri(apkPath);
            Intent intent = new Intent(Intent.ACTION_VIEW);
            intent.setDataAndType(installUri, "application/vnd.android.package-archive");
            String targetPkg = PkgUtils.queryDefaultPackageForIntent(intent);
            if (TextUtils.isEmpty(targetPkg)) {
                Log.e(TAG, "no activity found to install apk");
                return;
            }
            if (TextUtils.equals(installUri.getScheme(), "content")) {
                mContext.grantUriPermission(targetPkg, installUri, Intent.FLAG_GRANT_READ_URI_PERMISSION);
            }
            intent.setPackage(targetPkg);
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            mContext.startActivity(intent);
        }

        private boolean verify(String apkFilePath) {
            if (TextUtils.isEmpty(apkFilePath)) {
                return false;
            }
            String fileHash = Coder.encodeMD5(new File(apkFilePath));
            return TextUtils.equals(fileHash, mUpdateInfo.apkHash);
        }

        private String getPatchedApk(String diffFile, String diffFileHash) {
            if (!TextUtils.isEmpty(diffFileHash)) {
                String fileHash = Coder.encodeMD5(new File(diffFile));
                if (!TextUtils.equals(fileHash, diffFileHash)) {
                    return null;
                }
            }
            String apkPath = diffFile + ".apk";
            if (mAppInfo != null && !TextUtils.isEmpty(mAppInfo.sourceDir)) {
                Patcher.patch(mAppInfo.sourceDir, apkPath, diffFile);
                try {
                    new File(diffFile).delete();
                } catch(Exception e) {}
                return apkPath;
            }
            return null;
        }
    }

    private Uri generateInstallUri(String apkFilePath) {
        Uri installUri;
        if (Client.isLaterThanN()) {
            File file = new File(apkFilePath);
            String authority = mContext.getPackageName() + ".selfupdate.fileprovider";
            installUri = FileProvider.getUriForFile(mContext, authority, file);
        } else {
            installUri = Uri.parse("file://" + apkFilePath);
        }
        return installUri;
    }

    private static class DownloadManagerInfo {
        public long id;
        public int status;
        public int reason;
        public int currBytes;
        public int totalBytes;
        public String downloadFilePath;

        public static DownloadManagerInfo find(long id) {
            Query query = new Query();
            query.setFilterById(id);
            Cursor cursor;
            try {
                cursor = mDownloadManager.query(query);
            } catch (Exception e) {
                Log.e(TAG, "Query download from DownloadManager failed - " + e.toString());
                return null;
            }

            try {
                if ((cursor != null) && cursor.moveToFirst()) {
                    return query(cursor);
                }
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
            return null;
        }

        @SuppressLint("InlinedApi")
        private static DownloadManagerInfo query(Cursor cursor) {
            int idColumn = cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_ID);
            int statusColumn = cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_STATUS);
            int reasonColumn = cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_REASON);
            int currSizeColumn = cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_BYTES_DOWNLOADED_SO_FAR);
            int totalSizeColumn = cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_TOTAL_SIZE_BYTES);
            String filePathColumnName;
            if (Client.isLaterThanHoneycomb()) { // 这里不要判断MIUI，只判断Android版本
                filePathColumnName = DownloadManager.COLUMN_LOCAL_FILENAME;
            } else {
                filePathColumnName = COLUMN_FILE_PATH;
            }
            int localFileNameColumn = cursor.getColumnIndexOrThrow(filePathColumnName);

            DownloadManagerInfo info = new DownloadManagerInfo();
            info.id = cursor.getLong(idColumn);
            info.status = cursor.getInt(statusColumn);
            info.reason = cursor.getInt(reasonColumn);
            info.currBytes = cursor.getInt(currSizeColumn);
            info.totalBytes = cursor.getInt(totalSizeColumn);
            info.downloadFilePath = cursor.getString(localFileNameColumn);
            return info;
        }
    }
}
