
package com.xiaomi.market.sdk;

import android.text.TextUtils;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.TreeMap;

public class Connection {
    private static final String TAG = "MarketConnection";

    protected static final String PROTOCOL_HTTP = "http";
    protected static final String PROTOCOL_HTTPS = "https";
    private static final int CONNECT_TIMEOUT = 10000;
    private static final int WIFI_READ_TIMEOUT = 10000;
    private static final int GPRS_READ_TIMEOUT = 30000;

    public static enum NetworkError {
        OK,
        URL_ERROR,
        NETWORK_ERROR,
        AUTH_ERROR, // 在该链接需要登录但是没有的情况下返回
        CLIENT_ERROR,
        SERVER_ERROR,
        RESULT_ERROR,
        UNKNOWN_ERROR
    }

    protected JSONObject mResponse;
    protected URL mUrl;
    protected Parameter mParameter;
    protected String mString;

    protected boolean mNeedBaseParameter;
    protected boolean mUseGet;
    protected boolean mNeedHosted;
    protected boolean mNeedId;
    protected boolean mNeedSessionID;

    protected boolean mIsBackground;

    public Connection(String urlstring) {
        this(urlstring, false);
    }

    public Connection(String baseUrlString, String appendUrlString) {
        this(connect(baseUrlString, appendUrlString), false);
    }

    public Connection(String urlstring, boolean background) {
        URL url = null;
        try {
            url = new URL(urlstring);
        } catch (MalformedURLException e) {
            Log.e(TAG, "URL error: " + e);
        }
        init(url);

        mIsBackground = background;
    }

    public static String connect(String baseUrlString, String appendUrlString) {
        if (TextUtils.isEmpty(baseUrlString)) {
            return appendUrlString;
        }
        if (TextUtils.isEmpty(appendUrlString)) {
            return baseUrlString;
        }

        if (baseUrlString.charAt(baseUrlString.length() - 1) == '/') {
            baseUrlString = baseUrlString.substring(0, baseUrlString.length() - 1);
        }
        if (appendUrlString.charAt(0) == '/') {
            appendUrlString = appendUrlString.substring(1);
        }
        return baseUrlString + "/" + appendUrlString;
    }

    private void init(URL url) {
        mNeedBaseParameter = true;
        mUseGet = false;
        mNeedHosted = true;
        mNeedId = true;
        mNeedSessionID = true;
        if (checkURL(url)) {
            mUrl = url;
        }
    }

    public JSONObject getResponse() {
        return mResponse;
    }

    public String getStringResponse() {
        return mString;
    }

    public Parameter getParameter() {
        return mParameter;
    }

    public void setUseGet(boolean useGet) {
        mUseGet = useGet;
    }

    public void setNeedBaseParameter(boolean need) {
        mNeedBaseParameter = need;
    }

    public void setNeedHosted(boolean need) {
        mNeedHosted = need;
    }

    public void setNeedId(boolean need) {
        mNeedId = need;
    }

    public void setNeedSessionId(boolean need) {
        mNeedSessionID = need;
    }

    /**
     * 请求json，该方法必须在后台线程中调用
     */
    public NetworkError requestJSON() {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        NetworkError resp = request(new MemoryResetableOutputStream(baos));
        try {
            if (resp == NetworkError.OK) {
                mResponse = new JSONObject(baos.toString());
            } else {
                Log.e(TAG, "Connection failed : " + resp);
            }
        } catch (JSONException e) {
            Log.e(TAG, "JSON error: " + e);
            return NetworkError.RESULT_ERROR;
        } finally {
            try {
                baos.close();
            } catch (IOException e) {
            }
        }
        return resp;
    }

    public NetworkError requestString() {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        NetworkError resp = request(new MemoryResetableOutputStream(baos));
        if (resp == NetworkError.OK) {
            mString = baos.toString();
        } else {
            Log.e(TAG, "Connection failed : " + resp);
        }
        try {
            baos.close();
        } catch (IOException e) {
        }
        return resp;
    }

    /**
     * 请求文件，该方法必须在后台线程中调用
     */
    public NetworkError requestFile(File outFile) throws FileNotFoundException {
        if (outFile == null) {
            throw new IllegalArgumentException();
        }
        FileResetableOutputStream fos = null;
        try {
            fos = new FileResetableOutputStream(outFile);
        } catch (FileNotFoundException e) {
            Log.e(TAG, "File not found: " + e);
            throw e;
        }
        NetworkError resp = request(fos);
        try {
            fos.close();
            if (resp != NetworkError.OK) {
                Log.e(TAG, "Connection failed : " + resp);
                outFile.delete();
            }
        } catch (IOException e) {
        }
        return resp;
    }

    protected NetworkError request(ResetableOutputStream outputStream) {
        if (mUrl == null) {
            // url有问题
            return NetworkError.URL_ERROR;
        }

        if (!Utils.isConnected(XiaomiUpdateAgent.getContext())) {
            // 网络未链接，直接返回错误
            return NetworkError.NETWORK_ERROR;
        }

        if (mParameter == null) {
            // 如果用户没有指定参数，则加上基础参数
            mParameter = this.new Parameter();
        }

        // 处理参数，例如加密等
        Parameter finalParams = mParameter;
        try {
            finalParams = onQueryCreated(mParameter);
        } catch (ConnectionException e) {
            return e.mError;
        }

        // 构造url
        String url = mUrl.toString();
        if (mUseGet) {
            // get parameters
            if (!finalParams.isEmpty()) {
                String query = mUrl.getQuery();
                String urlString = mUrl.toString();
                if (TextUtils.isEmpty(query)) {
                    urlString = urlString + "?" + finalParams.toString();
                } else {
                    urlString = urlString + "&" + finalParams.toString();
                }
                url = urlString;
            }
        }

        // 处理链接，例如加上签名等
        try {
            url = onURLCreated(url, finalParams);
        } catch (ConnectionException e) {
            return e.mError;
        }

        if (Utils.DEBUG) {
            Log.d(TAG, "connection url: " + url);
        }

        // 如果是使用staging服务器的情况下，不要使用落地节点

        // post 数据
        String postData = "";
        if (!mUseGet) {
            postData = finalParams.toString();
        }

        long start_ms = System.currentTimeMillis();
        NetworkError err = innerRequest(url, postData, mUseGet, false, outputStream);
        if (Utils.DEBUG) {
            long end_ms = System.currentTimeMillis();
            Log.d(TAG, "Time(ms) spent in request: " + (end_ms - start_ms) + ", " + url);
        }
        return err;
    }

    private NetworkError innerRequest(String url, String postData, boolean useGet,
            boolean needHosted, ResetableOutputStream outputStream) {
        ArrayList<String> retryUrls;
        retryUrls = new ArrayList<String>();
        retryUrls.add(url);

        // 获取数据
        for (String retryUrl : retryUrls) {
            if (Utils.DEBUG) {
                Log.d(TAG, "hosted connection url: " + retryUrl);
            }
            HttpURLConnection conn = null;

            URL currUrl = null;
            try {
                currUrl = new URL(retryUrl);
            } catch (MalformedURLException e) {
                Log.e(TAG, " URL error :" + e);
                continue;
            }

            try {
                conn = (HttpURLConnection) currUrl.openConnection();
                conn.setConnectTimeout(CONNECT_TIMEOUT);
                if (Utils.isWifiConnected(XiaomiUpdateAgent.getContext())) {
                    conn.setReadTimeout(WIFI_READ_TIMEOUT);
                } else {
                    conn.setReadTimeout(GPRS_READ_TIMEOUT);
                }
                if (useGet) {
                    conn.setRequestMethod("GET");
                    conn.setDoOutput(false);
                } else {
                    conn.setRequestMethod("POST");
                    conn.setDoOutput(true);
                }
                try {
                    conn = onConnectionCreated(conn);
                } catch (ConnectionException e) {
                    return e.mError;
                }
                conn.connect();

                // post data
                if (!useGet && !TextUtils.isEmpty(postData)) {
                    OutputStream out = conn.getOutputStream();
                    out.write(postData.getBytes());
                    if (Utils.DEBUG) {
                        Log.d(TAG, "[post]" + postData);
                    }
                    out.close();
                }

                int responseCode = conn.getResponseCode();
                NetworkError code = handleResponseCode(responseCode);
                if (code == NetworkError.OK) {
                    if (outputStream != null) {
                        BufferedInputStream bis = null;
                        try {
                            bis = new BufferedInputStream(conn.getInputStream(), 8192);
                            byte[] buffer = new byte[1024];
                            int count;
                            while ((count = bis.read(buffer, 0, 1024)) > 0) {
                                outputStream.write(buffer, 0, count);
                            }
                            outputStream.flush();
                        } catch (Exception e) {
                            // 读取文件流的过程中异常有可能造成文件损坏，页面读取时造成ANR
                            Log.e(TAG, "Connection Exception for " + currUrl.getHost()
                                    + " : read file stream error " + e);
                            outputStream.reset(); // 重置输出流
                            continue;
                        } finally {
                            if (bis != null) {
                                bis.close();
                            }
                        }
                    }
                }
                // 执行到这里不会是网络问题，不再重试
                return code;
            } catch (Exception e) {
                Log.e(TAG, "Connection Exception for " + currUrl.getHost() + " :" + e);
            } finally {
                if (conn != null) {
                    conn.disconnect();
                }
            }
        }
        // 执行到这里说明网络有问题，一个都没有成功
        return NetworkError.NETWORK_ERROR;
    }

    /**
     * hook for subclasses to modify the parameters before connect, this method
     * will be called before {@code onURLCreated(String);}
     */
    protected Parameter onQueryCreated(Parameter params) throws ConnectionException {
        return params;
    }

    /**
     * hook for subclasses to modify url before connection created，
     * {@code finalParams} is the result of {@code onQueryCreated(Parameter);}
     */
    protected String onURLCreated(String url, Parameter finalParams) throws ConnectionException {
        return url;
    }

    /**
     * hook for subclasses to modify the connection before connect
     */
    protected HttpURLConnection onConnectionCreated(HttpURLConnection connection)
            throws ConnectionException {
        return connection;
    }

    protected boolean checkURL(URL url) {
        if (url == null) {
            return false;
        }
        String protocol = url.getProtocol();
        return TextUtils.equals(protocol, PROTOCOL_HTTP) || TextUtils.equals(protocol, PROTOCOL_HTTPS);
    }

    private NetworkError handleResponseCode(int code) {
        if (code == HttpURLConnection.HTTP_OK) {
            return NetworkError.OK;
        } else {
            Log.e(TAG, "Network Error : " + code);
            return NetworkError.SERVER_ERROR;
        }
    }

    public class Parameter {
        // 参数是有序的，按照string的比较顺序
        private TreeMap<String, String> params;

        /**
         * 用户使用当前对象的构造方法时与相对应的Connection绑定.<br>
         * 例如：<br>
         * Connection conn = new Connection();<br>
         * Parameter para = conn.new Parameter();<br>
         * 返回的para已经被加载到conn中
         */
        public Parameter() {
            this(true);
        }

        public Parameter(boolean bindToConnection) {
            params = new TreeMap<String, String>();
            if (bindToConnection) {
                // 参数是否要绑定到当前链接上
                Connection.this.mParameter = this;
            }
        }

        public Parameter add(String key, String value) {
            if (value == null) {
                value = "";
            }
            params.put(key, value);
            return this;
        }

        public Parameter add(String key, boolean value) {
            if (value) {
                params.put(key, "true");
            } else {
                params.put(key, "false");
            }
            return this;
        }

        public String get(String key) {
            return params.get(key);
        }

        public boolean isEmpty() {
            return params.isEmpty();
        }

        public String toString() {
            if (params.isEmpty()) {
                return "";
            }
            StringBuilder sb = new StringBuilder();
            for (String key : params.keySet()) {
                sb.append(key);
                sb.append("=");
                try {
                    sb.append(URLEncoder.encode(params.get(key), "UTF-8"));
                } catch (UnsupportedEncodingException e) {
                }
                sb.append("&");
            }
            return sb.deleteCharAt(sb.length() - 1).toString();
        }

        public TreeMap<String, String> getParams() {
            return params;
        }
    }

    protected class ConnectionException extends Exception {
        /**
         *
         */
        private static final long serialVersionUID = 1L;
        protected NetworkError mError;

        public ConnectionException(NetworkError error) {
            mError = error;
        }
    }

    protected abstract class ResetableOutputStream extends OutputStream {
        protected OutputStream mOutputStream;

        public ResetableOutputStream(OutputStream outputStream) {
            if (outputStream == null) {
                throw new IllegalArgumentException("outputstream is null");
            }
            mOutputStream = outputStream;
        }

        @Override
        public void close() throws IOException {
            mOutputStream.close();
        }

        @Override
        public void flush() throws IOException {
            mOutputStream.flush();
        }

        @Override
        public void write(byte[] buffer) throws IOException {
            mOutputStream.write(buffer);
        }

        @Override
        public void write(byte[] buffer, int offset, int count) throws IOException {
            mOutputStream.write(buffer, offset, count);
        }

        @Override
        public void write(int oneByte) throws IOException {
            mOutputStream.write(oneByte);
        }

        public abstract void reset();
    }

    protected class MemoryResetableOutputStream extends ResetableOutputStream {

        public MemoryResetableOutputStream(ByteArrayOutputStream outputStream) {
            super(outputStream);
        }

        @Override
        public void reset() {
            ((ByteArrayOutputStream) mOutputStream).reset();
        }
    }

    protected class FileResetableOutputStream extends ResetableOutputStream {
        private File mFile;

        public FileResetableOutputStream(File file) throws FileNotFoundException {
            super(new FileOutputStream(file));
            mFile = file;
        }

        @Override
        public void reset() {
            try {
                mOutputStream.close();
            } catch (IOException e) {
                // do nothing
            }
            mFile.delete();
            try {
                mOutputStream = new FileOutputStream(mFile);
            } catch (FileNotFoundException e) {
                // 这里不会执行到，构造方法会抛出异常
            }
        }
    }
}
