package litifer.com.sdk.presentation;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.nfc.Tag;
import android.os.Build;
import android.os.IBinder;
import android.os.RemoteException;
import android.os.SystemClock;
import android.support.annotation.IntDef;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.util.Log;
import android.util.Pair;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.MonitorNotifier;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;

import litifer.com.sdk.domain.executor.impl.JobExecutor;
import litifer.com.sdk.presentation.presenters.BeaconServicePresenter;
import litifer.com.sdk.presentation.subscriber.thread.UIThread;

public class BeaconService extends Service implements BeaconConsumer{

    private static final String TAG = "BeaconsEverywhere";

    private BeaconManager beaconManager;


    private boolean isAppForeground;
    private long stayUnitDuration;
    private Region regionToMonitor=new Region("myMonitoringUniqueId", null,null,null);
//    private Collection<Beacon> mActiveBeacons;
    HashMap<Beacon,Pair<Integer,Long>> mActiveBeacons=new HashMap<Beacon, Pair<Integer,Long>>();



    //presenters

    private BeaconServicePresenter beaconServicePresenter;

    //event monitoring
    public enum TRIGGER{
        ENTRY(0),
        EXIT(5),
        TRIGGER_DURATION(60000);

        private int intValue;
        private TRIGGER(int intValue){
            this.intValue=intValue;
        }

    }
    // unused variables and functions that could be used later
    // private ArrayList<Region> mRegionList;
    // private IBinder mBinder = new MyBinder();


    //testing
    private long prevtime=System.currentTimeMillis();


    public BeaconService(){
        super();
    }


    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN_MR2)
    @Override
    public void onCreate() {
        super.onCreate();

        Log.d(TAG,"Started");
        //******************Creating Instance of BeaconManager ******************
        beaconManager = BeaconManager.getInstanceForApplication(this);
        setup();
        //        BeaconManager.setRegionExitPeriod(20001);

        /**
         * when mobile is in beacon range it does scanning/ranging at foreground scanning rate.
         */
        stayUnitDuration = BeaconManager.DEFAULT_FOREGROUND_BETWEEN_SCAN_PERIOD + BeaconManager.DEFAULT_FOREGROUND_SCAN_PERIOD;
        beaconServicePresenter =new BeaconServicePresenter(new JobExecutor(),UIThread.getThread(),this);

    }




    private void setup(){
        //******************Setting Beacon Layout.It is specific to beacon manufacturer.***************
        try {
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(BeaconParser.ALTBEACON_LAYOUT));
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout("m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24"));
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(BeaconParser.URI_BEACON_LAYOUT));
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(BeaconParser.EDDYSTONE_UID_LAYOUT));
            beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(BeaconParser.EDDYSTONE_TLM_LAYOUT));
        }catch(Exception e){
            e.printStackTrace();
        }

        beaconManager.bind(this);
    }



    //*******************This Region is only useful if BeaconManager and Beaconlayout are set appropriately***************
    //*******************So basically this when app connects to beacon****************************************************
    @Override
    public void onBeaconServiceConnect() {


        //**************************Monitoring,Can know when enters and Exit.**********************************
        beaconManager.addMonitorNotifier(new MonitorNotifier() {
            @Override
            public void didEnterRegion(Region region) {
                Log.d(TAG, "didEnterRegion");
                try {
                    beaconManager.startRangingBeaconsInRegion(region);
                } catch (RemoteException e) {
                    e.printStackTrace();
                }
            }

            @Override
            public void didExitRegion(Region region) {
                Log.d(TAG, "didExitRegion");
                try {
                    beaconManager.stopRangingBeaconsInRegion(region);
                } catch (RemoteException e) {
                    e.printStackTrace();
                }
            }

            @Override
            public void didDetermineStateForRegion(int i, Region region) {
                System.out.println(i);
            }
        });


        //**************************Ranging,can calculate the distance. **********************************
        beaconManager.addRangeNotifier(new RangeNotifier() {
            @Override
            public void didRangeBeaconsInRegion(Collection<Beacon> beacons, Region region) {
                // copying it using hashset since shallow copying leads to pass by reference.
                Collection<Beacon> exitedBeacons = new HashSet<Beacon>(mActiveBeacons.keySet());
                Collection<Beacon> enteredBeacons= new HashSet<Beacon>(beacons);
                Collection<Beacon> retainedbeacons = new HashSet<Beacon>(mActiveBeacons.keySet());
                exitedBeacons.removeAll(beacons);
                enteredBeacons.removeAll(mActiveBeacons.keySet());
                retainedbeacons.retainAll(beacons);

                for (Beacon beacon:enteredBeacons){
                    mActiveBeacons.put(beacon,new Pair<>(TRIGGER.ENTRY.intValue,0l));
                    beaconServicePresenter.enterEvent(beacon);
                }

                for (Beacon beacon:exitedBeacons){
                    int triggerValue;
                    long durationStay = mActiveBeacons.get(beacon).second + stayUnitDuration;
                    triggerValue = mActiveBeacons.get(beacon).first + 1;


                    /**
                     * keep it above the beacon_exit check because it will add beacons to mActiveBeacons if this line is run
                     * after if-else block.
                     */

                mActiveBeacons.put(beacon, new Pair<>(triggerValue,durationStay));

                if (triggerValue == TRIGGER.EXIT.intValue) {
                    mActiveBeacons.remove(beacon);
                    beaconServicePresenter.exitEvent(beacon,durationStay);
                }else if (mActiveBeacons.get(beacon).second/TRIGGER.TRIGGER_DURATION.intValue != durationStay/TRIGGER.TRIGGER_DURATION.intValue){
                    beaconServicePresenter.delayEvent(beacon,durationStay);
                }
                }

                for (Beacon beacon:retainedbeacons){
                    long durationStay = mActiveBeacons.get(beacon).second + stayUnitDuration;

                    if (mActiveBeacons.get(beacon).second/TRIGGER.TRIGGER_DURATION.intValue != durationStay/TRIGGER.TRIGGER_DURATION.intValue){
                        beaconServicePresenter.delayEvent(beacon,durationStay);
                    }
                    mActiveBeacons.put(beacon,new Pair<>(TRIGGER.ENTRY.intValue,durationStay));

                }


                }

//                bluetoothCrashResolver.notifyScannedDevice(device, myLeScanCallback);

        });



        if (beaconManager.getMonitoredRegions().size()>0){
            try {
                beaconManager.stopMonitoringBeaconsInRegion(regionToMonitor);
            } catch (RemoteException e) {
                e.printStackTrace();
            }
        }

        try {
            beaconManager.startMonitoringBeaconsInRegion(regionToMonitor);
        } catch (RemoteException e) {
            e.printStackTrace();
        }


        Log.d(TAG,beaconManager.getRangedRegions().toString());

    }





    @Override
    public void onDestroy() {
        super.onDestroy();
        Log.d(TAG,"destroyed");
        beaconManager.unbind(this);
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }


    ///////////////////////////RESTART BEACON SERVICE ONCE IT CLOSES////////////////////////////////
    @Override
    public void onTaskRemoved(Intent rootIntent){
        Intent restartServiceIntent = new Intent(getApplicationContext(), this.getClass());
        restartServiceIntent.setPackage(getPackageName());

        PendingIntent restartServicePendingIntent = PendingIntent.getService(getApplicationContext(), 1, restartServiceIntent, PendingIntent.FLAG_ONE_SHOT);

        AlarmManager alarmService = (AlarmManager) getApplicationContext().getSystemService(Context.ALARM_SERVICE);
        alarmService.set(
                AlarmManager.ELAPSED_REALTIME,
                SystemClock.elapsedRealtime() + 1000,
                restartServicePendingIntent);
        Log.d(TAG,"ontask beacon");
        super.onTaskRemoved(rootIntent);

    }

    @Override
    public int onStartCommand(Intent intent,int flags, int startId) {
        Log.d(TAG,"On Start beacons");

       return START_STICKY;
    }
}
