package litifer.com.sdk.domain.interactors.base;



import android.content.Context;

import java.io.IOException;

import litifer.com.sdk.presentation.subscriber.thread.PostExecutionThread;
import litifer.com.sdk.domain.executor.ThreadExecutor;
import rx.Observable;
import rx.Subscriber;
import rx.Subscription;
import rx.schedulers.Schedulers;
import rx.subscriptions.Subscriptions;

/**
 * Created by kishlaykishore on 07/03/17.
 * This abstract class implements some common methods for all interactors. Cancelling an interactor, check if its running
 * and finishing an interactor has mostly the same code throughout so that is why this class was created. Field methods
 * are declared volatile as we might use these methods from different threads (mainly from UI).
 * <p/>
 * For example, when an activity is getting destroyed then we should probably cancel an interactor
 * but the request will come from the UI thread unless the request was specifically assigned to a background thread.
 */

public abstract class AbstractInteractor implements Interactor{
    protected final ThreadExecutor threadExecutor;
    protected final PostExecutionThread postExecutionThread;
    protected Context context;



    private Subscription subscription = Subscriptions.empty();

    protected AbstractInteractor(ThreadExecutor threadExecutor,
                                 PostExecutionThread postExecutionThread,Context context) {
        this.threadExecutor = threadExecutor;
        this.postExecutionThread = postExecutionThread;
        this.context=context;
    }

    /**
     * Builds an {@link rx.Observable} which will be used when executing the current {@link }.
     */
    protected abstract Observable buildUseCaseObservable() throws IOException;


    /**
     * Executes the current use case.
     *
     * @param useCaseSubscriber The guy who will be listen to the observable build
     * with {@link #buildUseCaseObservable()}.
     */
    @SuppressWarnings("unchecked")
    @Override
    public void execute(Subscriber useCaseSubscriber) {
        try {
            this.subscription = this.buildUseCaseObservable()
                    .subscribeOn(Schedulers.from(threadExecutor))
                    .observeOn(postExecutionThread.getScheduler())
                    .subscribe(useCaseSubscriber);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Unsubscribes from current {@link rx.Subscription}.
     */
    public void unsubscribe() {
        if (!subscription.isUnsubscribed()) {
            subscription.unsubscribe();
        }
    }

    /**
     * This method contains the actual business logic of the interactor. It SHOULD NOT BE USED DIRECTLY but, instead, a
     * developer should call the execute() method of an interactor to make sure the operation is done on a background thread.
     * <p/>
     * This method should only be called directly while doing unit/integration tests. That is the only reason it is declared
     * public as to help with easier testing.
     */



//    protected volatile boolean mIsCanceled;
//    protected volatile boolean mIsRunning;
//    public abstract void run();
//
//    public void cancel() {
//        mIsCanceled = true;
//        mIsRunning = false;
//    }
//
//    public boolean isRunning() {
//        return mIsRunning;
//    }
//
//    public void onFinished() {
//        mIsRunning = false;
//        mIsCanceled = false;
//    }
//
//    public void execute() {
//
//        // mark this interactor as running
//        this.mIsRunning = true;
//
//        // start running this interactor in a background thread
//        mThreadExecutor.execute(this);
//    }

}
