/*
 * Copyright (c) 2022 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */
package ee.jakarta.tck.data.framework.junit.extensions;

import java.lang.reflect.Method;
import java.util.Optional;
import java.util.logging.Logger;

import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.api.extension.TestWatcher;

import ee.jakarta.tck.data.framework.junit.anno.Assertion;

/**
 * If a test fails or is disabled we can warn the implementor of the assertion
 * ID As well as provide test strategy information to aid in resolving the
 * issue. This data is obtained by the {@code @Assertion} annotation
 * 
 * @see ee.jakarta.tck.data.framework.junit.anno.Assertion
 */
public class AssertionExtension implements TestWatcher {
    private static final Logger log = Logger.getLogger(AssertionExtension.class.getCanonicalName());
    private static final String nl = System.lineSeparator();

    @Override
    public void testFailed(ExtensionContext context, Throwable cause) {
        Method testMethod = context.getRequiredTestMethod();
        Assertion instance = testMethod.getAnnotation(Assertion.class);
        if (instance != null) {
            log.warning(testMethod.getName() + " failed " + nl
                    + " @Assertion.id:" + instance.id() + nl
                    + " @Assertion.strategy: " + instance.strategy());
        }
    }

    @Override
    public void testAborted(ExtensionContext context, Throwable cause) {
        Method testMethod = context.getRequiredTestMethod();
        Assertion instance = testMethod.getAnnotation(Assertion.class);
        if (instance != null) {
            log.warning(testMethod.getName() + " was aborted " + nl
                    + " @Assertion.id:" + instance.id() + nl
                    + " @Assertion.strategy: " + instance.strategy());
        }
    }

    @Override
    public void testDisabled(ExtensionContext context, Optional<String> reason) {
        Method testMethod = context.getRequiredTestMethod();
        Assertion instance = testMethod.getAnnotation(Assertion.class);
        if (instance != null) {
            log.warning(testMethod.getName() + " is disabled" + nl
                    + " @Assertion.id:" + instance.id() + nl
                    + " @Assertion.strategy: " + instance.strategy() + nl
                    + " Reason:" + reason.get());
        }
    }
}
