/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.util;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.uber.tchannel.codecs.CodecUtils;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.EmptyByteBuf;
import io.netty.buffer.Unpooled;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

public class HeaderUtil {

    public static Map<String, String> decodeHeaders(ByteBuf headers) {
        if (headers == null || headers instanceof EmptyByteBuf) {
            return new HashMap<>();
        }
        return CodecUtils.decodeHeaders(headers);
    }

    public static Map<String, String> decodeHeadersJsonStyle(ByteBuf headers) throws IOException {
        if (headers == null || headers instanceof EmptyByteBuf) {
            return new HashMap<>();
        }

        ObjectMapper mapper = new ObjectMapper();
        return mapper.readValue(
                BufferUtil.toBytes(headers),
                mapper.getTypeFactory().constructMapType(HashMap.class, String.class, String.class)
        );
    }

    public static ByteBuf encodeHeaders(Map<String, String> applicationHeaders) {
        ByteBuf buf = ByteBufAllocator.DEFAULT.buffer();
        CodecUtils.encodeHeaders(applicationHeaders, buf);
        return buf;
    }

    public static ByteBuf encodeHeadersJsonStyle(Map<String, String> applicationHeaders) throws
            JsonProcessingException {
        ObjectMapper objectMapper = new ObjectMapper();
        return Unpooled.wrappedBuffer(objectMapper.writeValueAsBytes(applicationHeaders));
    }
}
