/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.transport.tchannel;

import com.uber.tchannel.api.SubChannel;
import com.uber.tchannel.api.TChannel;
import com.uber.tchannel.api.TFuture;
import com.uber.tchannel.api.errors.TChannelError;
import com.uber.tchannel.messages.JsonRequest;
import com.uber.tchannel.messages.JsonResponse;
import com.uber.tchannel.messages.RawRequest;
import com.uber.tchannel.messages.RawResponse;
import com.uber.tchannel.messages.ThriftRequest;
import com.uber.tchannel.messages.ThriftResponse;
import io.netty.buffer.Unpooled;
import io.yarpc.Request;
import io.yarpc.Response;
import io.yarpc.encoding.Encoding;
import io.yarpc.exception.RPCException;
import io.yarpc.transport.Outbound;
import io.yarpc.transport.ResponseListener;
import io.yarpc.util.HeaderUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.util.concurrent.ExecutionException;

/**
 * TChannel outbound supports raw, json, thrift encoding.
 */
public class TChannelOutbound extends Outbound {
    private static final Logger logger = LoggerFactory.getLogger(TChannelOutbound.class);
    private final TChannel tchannel;
    private InetSocketAddress host;

    public TChannelOutbound(TChannel tchannel, String ip, int port) throws IOException {
        this.host = new InetSocketAddress(ip, port);
        this.tchannel = tchannel;
    }

    @Override
    public void send(Request<ByteBuffer> transportRequest, ResponseListener responseListener) throws RPCException {
        SubChannel subChannel = this.tchannel.makeSubChannel(transportRequest.getService());
        try {
            this.sendRequest(subChannel, transportRequest, responseListener);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void stop() {
    }

    private void sendRequest(SubChannel subChannel, Request<ByteBuffer> transportRequest, ResponseListener
            responseListener)
            throws Exception {
        switch (transportRequest.getEncoding()) {
            case Encoding.JSON:
                this.sendJsonRequest(subChannel, transportRequest, responseListener);
                break;
            case Encoding.THRIFT:
                this.sendThriftRequest(subChannel, transportRequest, responseListener);
                break;
            case Encoding.RAW:
                this.sendRawRequest(subChannel, transportRequest, responseListener);
                break;
            default:
                throw new RPCException("Encoding type is not supported in TChannel protocol.");
        }
    }

    private void sendThriftRequest(
            SubChannel subChannel, Request<ByteBuffer> transportRequest, ResponseListener responseListener) throws
            InterruptedException, ExecutionException, TChannelError {
        ThriftRequest req = new ThriftRequest.Builder(
                transportRequest.getService(),
                transportRequest.getProcedure())
                .setArg3(Unpooled.wrappedBuffer(transportRequest.getBody()))
                .setHeaders(transportRequest.getHeaders())
                .setTimeout(transportRequest.getTimeout().getMillis())
                .build();
        TFuture<ThriftResponse> future = subChannel.send(req, this.host.getAddress(), this.host.getPort());
        ThriftResponse response = future.get();

        if (response.isError()) {
            Response<ByteBuffer> transportResponse = new Response.Builder<ByteBuffer>()
                    .exception(new RPCException(response.getError().getMessage()))
                    .headers(response.getHeaders())
                    .build();
            responseListener.onErrorReceived(transportResponse);
        } else {
            Response<ByteBuffer> transportResponse = new Response.Builder<ByteBuffer>()
                    .body(response.getArg3().nioBuffer())
                    .headers(response.getHeaders())
                    .build();
            responseListener.onResponseReceived(transportResponse);
        }
    }

    private void sendJsonRequest(
            SubChannel subChannel, Request<ByteBuffer> transportRequest, ResponseListener responseListener) throws
            InterruptedException, ExecutionException {
        JsonRequest req = new JsonRequest.Builder(
                transportRequest.getService(),
                transportRequest.getProcedure())
                .setArg3(Unpooled.wrappedBuffer(transportRequest.getBody()))
                .setHeaders(transportRequest.getHeaders())
                .setTimeout(transportRequest.getTimeout().getMillis())
                .build();
        TFuture<JsonResponse<Object>> future = subChannel.send(req, this.host.getAddress(), this.host.getPort());
        JsonResponse response = future.get();

        if (response.isError()) {
            Response<ByteBuffer> transportResponse = new Response.Builder<ByteBuffer>()
                    .exception(new RPCException(response.getError().getMessage()))
                    .headers(response.getHeaders())
                    .build();
            responseListener.onErrorReceived(transportResponse);
        } else {
            Response<ByteBuffer> transportResponse = new Response.Builder<ByteBuffer>()
                    .body(response.getArg3().nioBuffer())
                    .headers(response.getHeaders())
                    .build();
            responseListener.onResponseReceived(transportResponse);
        }
    }

    private void sendRawRequest(
            SubChannel subChannel, Request<ByteBuffer> transportRequest, ResponseListener responseListener) throws
            InterruptedException, ExecutionException, IOException {
        RawRequest req = new RawRequest.Builder(
                transportRequest.getService(),
                transportRequest.getProcedure())
                .setArg3(Unpooled.wrappedBuffer(transportRequest.getBody()))
                .setArg2(Unpooled.wrappedBuffer(HeaderUtil.encodeHeaders(transportRequest.getHeaders())))
                .setTimeout(transportRequest.getTimeout().getMillis())
                .build();
        TFuture<RawResponse> future = subChannel.send(req, this.host.getAddress(), this.host.getPort());
        RawResponse response = future.get();

        if (response.isError()) {
            Response<ByteBuffer> transportResponse = new Response.Builder<ByteBuffer>()
                    .exception(new RPCException(response.getError().getMessage()))
                    .headers(HeaderUtil.decodeHeaders(response.getArg2()))
                    .build();
            responseListener.onErrorReceived(transportResponse);
        } else {
            Response<ByteBuffer> transportResponse = new Response.Builder<ByteBuffer>()
                    .body(response.getArg3().nioBuffer())
                    .headers(HeaderUtil.decodeHeaders(response.getArg2()))
                    .build();
            responseListener.onResponseReceived(transportResponse);
        }
    }
}
