/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.transport.tchannel;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.uber.tchannel.api.handlers.RequestHandler;
import com.uber.tchannel.headers.ArgScheme;
import com.uber.tchannel.messages.RawResponse;
import com.uber.tchannel.messages.Request;
import com.uber.tchannel.messages.Response;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.yarpc.HandlerRegistry;
import io.yarpc.exception.RPCException;
import io.yarpc.handler.HandlerWrapper;
import io.yarpc.util.HeaderUtil;
import org.joda.time.Duration;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;

public class TChannelInboundHandler implements RequestHandler {
    private final HandlerRegistry handlerRegistry;

    public TChannelInboundHandler(HandlerRegistry handlerRegistry) {
        this.handlerRegistry = handlerRegistry;
    }

    private io.yarpc.Request<ByteBuffer> parseRequest(Request tchannelRequest) throws IOException {
        Map<String, String> headers;
        if (tchannelRequest.getArgScheme() == ArgScheme.JSON) {
            headers = HeaderUtil.decodeHeadersJsonStyle(tchannelRequest.getArg2());
        } else {
            headers = HeaderUtil.decodeHeaders(tchannelRequest.getArg2());
        }
        io.yarpc.Request<ByteBuffer> transportRequest = new io.yarpc.Request.Builder<ByteBuffer>()
                .body(tchannelRequest.getArg3().nioBuffer())
                .headers(headers)
                .procedure(tchannelRequest.getEndpoint())
                .service(tchannelRequest.getService())
                // TODO implement caller
                .timeout(new Duration(tchannelRequest.getTimeout()))
                .build();
        return transportRequest;
    }

    @Override
    public Response handle(Request request) {
        try {
            HandlerWrapper handlerWrapper = this.validateRequest(request);
            // TODO build yarpc common context including tracing, meta info.
            // Context context = ContextBuilder.build(httpRequest);

            io.yarpc.Request<ByteBuffer> transportRequest = this.parseRequest(request);
            io.yarpc.Response<ByteBuffer> transportResponse = handlerWrapper.handle(transportRequest);

            ByteBuf headers;
            if (request.getArgScheme() == ArgScheme.JSON) {
                headers = HeaderUtil.encodeHeadersJsonStyle(transportResponse.getHeaders());
            } else {
                headers = HeaderUtil.encodeHeaders(transportResponse.getHeaders());
            }
            RawResponse response = new RawResponse.Builder(request)
                    .setArg2(headers)
                    .setArg3(Unpooled.wrappedBuffer(transportResponse.getBody()))
                    .build();
            return response;
        } catch (RPCException e) {
            // TODO log error
        } catch (JsonProcessingException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    private HandlerWrapper validateRequest(Request request) throws RPCException {
        // look up user procedure.
        String procedure = request.getEndpoint();
        HandlerWrapper handlerWrapper = this.handlerRegistry.lookupHandler(procedure);
        if (handlerWrapper == null) {
            throw new RPCException(String.format("Can't find handler for procedure: %s.", procedure));
        }

        return handlerWrapper;
    }
}
