/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.transport.http;

import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.DefaultHttpHeaders;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaderValues;
import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.handler.codec.http.HttpVersion;
import io.yarpc.HandlerRegistry;
import io.yarpc.Request;
import io.yarpc.Response;
import io.yarpc.context.Names;
import io.yarpc.exception.RPCException;
import io.yarpc.handler.HandlerWrapper;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;


public class HTTPServerInboundHandler extends SimpleChannelInboundHandler<FullHttpRequest> {
    private static final Logger logger = LoggerFactory.getLogger(HTTPServerInboundHandler.class);
    private final HandlerRegistry handlerRegistry;

    public HTTPServerInboundHandler(HandlerRegistry handlerRegistry) {
        this.handlerRegistry = handlerRegistry;
    }

    @Override
    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        ctx.fireChannelActive();
    }

    @Override
    protected void messageReceived(ChannelHandlerContext channelHandlerContext, FullHttpRequest httpRequest) throws
            Exception {
        HandlerWrapper handlerWrapper = this.validateRequest(httpRequest);
        this.forwardRequest(channelHandlerContext, httpRequest, handlerWrapper);
    }

    private Request<ByteBuffer> parseRequest(FullHttpRequest httpRequest) {
        Map<String, String> headers = new HashMap<>();
        for (Map.Entry<CharSequence, CharSequence> entry : httpRequest.headers().entries()) {
            headers.put(entry.getKey().toString(), entry.getValue().toString());
        }

        HttpHeaders httpHeaders = httpRequest.headers();
        Request<ByteBuffer> transportRequest = new Request.Builder<ByteBuffer>()
                .body(httpRequest.content().nioBuffer())
                .headers(headers)
                .procedure(httpHeaders.get(Names.PROCEDURE).toString())
                .service(httpHeaders.get(Names.SERVICE).toString())
                .caller(httpHeaders.get(Names.CALLER).toString())
                .timeout(new Duration(Long.valueOf(httpHeaders.get(Names.TTLMS).toString())))
                .build();
        return transportRequest;
    }

    private void forwardRequest(ChannelHandlerContext channelHandlerContext, FullHttpRequest httpRequest, HandlerWrapper
            handlerWrapper) throws RPCException {

        HttpHeaders httpHeaders = new DefaultHttpHeaders();

        // TODO build yarpc common context including tracing, meta info.
        // Context context = ContextBuilder.build(httpRequest);

        Request<ByteBuffer> transportRequest = this.parseRequest(httpRequest);
        Response<ByteBuffer> transportResponse = handlerWrapper.handle(transportRequest);

        FullHttpResponse httpResponse = new DefaultFullHttpResponse(
                HttpVersion.HTTP_1_1,
                HttpResponseStatus.OK,
                Unpooled.wrappedBuffer(transportResponse.getBody())
        );

        Map<String, String> responseHeaders = transportResponse.getHeaders();
        for (Map.Entry<String, String> entry : responseHeaders.entrySet()) {
            httpHeaders.set(entry.getKey(), entry.getValue());
        }

        httpHeaders.set(HttpHeaderNames.CONTENT_LENGTH, String.valueOf(httpResponse.content().readableBytes()));
        httpHeaders.set(HttpHeaderNames.CONNECTION, HttpHeaderValues.CLOSE);
        httpResponse.headers().set(httpHeaders);

        channelHandlerContext.writeAndFlush(httpResponse);
        channelHandlerContext.channel().close();
        channelHandlerContext.close();
    }

    private HandlerWrapper validateRequest(FullHttpRequest rawRequest) throws RPCException {
        // look up user procedure.
        String procedure = rawRequest.headers().get(Names.PROCEDURE).toString();
        HandlerWrapper handlerWrapper = this.handlerRegistry.lookupHandler(procedure);
        if (handlerWrapper == null) {
            throw new RPCException(String.format("Can't find handler for procedure: %s.", procedure));
        }

        return handlerWrapper;
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        // Close the connection when an exception is raised.
        logger.warn("Unexpected exception occurs: {}", cause);

        // TODO send error message back.
        ctx.channel().close();
        ctx.close();
    }
}
