/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.transport.http;


import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpVersion;
import io.yarpc.Request;
import io.yarpc.context.Names;
import io.yarpc.exception.RPCException;
import io.yarpc.transport.Outbound;
import io.yarpc.transport.ResponseListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;

/**
 * Http outbound supports http 1.1 protocol.
 */
public class HTTPOutbound extends Outbound {
    private static final Logger logger = LoggerFactory.getLogger(HTTPOutbound.class);
    private final static String PATH = "/yarpc/v1";
    private Bootstrap bootstrap;
    private InetSocketAddress host;
    private NioEventLoopGroup workerGroup = new NioEventLoopGroup();

    public HTTPOutbound(String ip, int port) throws IOException {
        this.host = new InetSocketAddress(ip, port);
        try {
            this.bootstrap = this.bootstrap(this.workerGroup, new HTTPClientInitializer());
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void send(Request<ByteBuffer> transportRequest, ResponseListener responseListener) throws RPCException {
        try {
            Channel channel = this.bootstrap.connect(this.host).sync().channel();
            channel.pipeline().addLast("custom", new HTTPClientInboundHandler(responseListener));
            this.sendRequest(channel, transportRequest);
            channel.closeFuture().addListener(ChannelFutureListener.CLOSE);
        } catch (Exception e) {
            throw new RPCException(String.format("request failed due to: %s", e.getMessage()));
        }
    }

    @Override
    public void stop() {
        this.workerGroup.shutdownGracefully();
    }

    private void sendRequest(Channel channel, Request<ByteBuffer> transportRequest) throws Exception {
        FullHttpRequest httpRequest = new DefaultFullHttpRequest(
                HttpVersion.HTTP_1_1,
                HttpMethod.POST,
                PATH);

        ByteBuf bbuf = Unpooled.wrappedBuffer(transportRequest.getBody());

        //TODO convert context property into request headers
        httpRequest.headers().add(Names.PROCEDURE, transportRequest.getProcedure());
        httpRequest.headers().add(Names.ENCODING, transportRequest.getEncoding());
        httpRequest.headers().add(Names.TTLMS, String.valueOf(transportRequest.getTimeout().getMillis()));
        httpRequest.headers().add(Names.CALLER, transportRequest.getCaller());
        httpRequest.headers().add(Names.SERVICE, transportRequest.getService());

        //HTTP standard request
        httpRequest.headers().add(HttpHeaderNames.CONTENT_LENGTH, String.valueOf(bbuf.readableBytes()));
        httpRequest.headers().add(HttpHeaderNames.HOST, this.host.getHostString());
        httpRequest.content().clear().writeBytes(bbuf);
        channel.writeAndFlush(httpRequest);
    }

    private Bootstrap bootstrap(EventLoopGroup eventLoopGroup, HTTPClientInitializer httpClientInitializer) throws
            InterruptedException {
        Bootstrap bootstrap = new Bootstrap();
        bootstrap.group(eventLoopGroup);
        bootstrap.channel(NioSocketChannel.class);
        bootstrap.handler(httpClientInitializer);
        return bootstrap;
    }
}
