/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.transport.http;

import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.FullHttpResponse;
import io.yarpc.Response;
import io.yarpc.context.ContextBuilder;
import io.yarpc.transport.ResponseListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;


public class HTTPClientInboundHandler extends SimpleChannelInboundHandler<FullHttpResponse> {
    private static final Logger logger = LoggerFactory.getLogger(HTTPClientInboundHandler.class);
    private final ResponseListener responseListener;

    public HTTPClientInboundHandler(ResponseListener responseListener) {
        this.responseListener = responseListener;
    }

    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        ctx.fireChannelActive();
    }

    @Override
    protected void messageReceived(ChannelHandlerContext channelHandlerContext, FullHttpResponse httpResponse) throws
            Exception {
        // TODO Convert to rpc context
        Map<String, String> headers = new HashMap<>();

        for (Map.Entry<CharSequence, CharSequence> entry : httpResponse.headers().entries()) {
            headers.put(entry.getKey().toString(), entry.getValue().toString());
        }

        Response<ByteBuffer> transportResponse = new Response.Builder<ByteBuffer>()
                .headers(headers)
                .context(ContextBuilder.build(httpResponse)) // TODO fix context
                .body(httpResponse.content().nioBuffer())
                .build();
        this.responseListener.onResponseReceived(transportResponse);
    }


    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        // Close the connection when an exception is raised.
        logger.warn("Unexpected exception occurs: {}", cause.getMessage());
        // TODO convert cause into TransportRepsonse
        //this.responseListener.onErrorReceived();
        ctx.close();
    }
}
