/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.handler;

import com.fasterxml.classmate.ResolvedType;
import com.fasterxml.classmate.TypeResolver;
import io.yarpc.Request;
import io.yarpc.Response;
import io.yarpc.encoding.Serializer;
import io.yarpc.exception.RPCException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;
import java.util.List;

public abstract class BaseInboundHandler implements HandlerWrapper {
    private static final Logger logger = LoggerFactory.getLogger(BaseInboundHandler.class);
    protected Serializer serializer;
    private Handler handler;
    private Class<?> requestBodyType;
    private Class<?> responseBodyType;

    public BaseInboundHandler(Serializer serializer, Handler handler) {
        this.parseHandler(handler);
        this.serializer = serializer;
    }

    @Override
    public Response<ByteBuffer> handle(Request<ByteBuffer> transportRequest) throws RPCException {
        this.validate(transportRequest);
        try {
            Request request = Request.fromByteBufferRequest(transportRequest, this.serializer, requestBodyType);
            Response response = this.getHandler().handle(request);
            return Response.toByteBufferResponse(response, this.serializer);
        } catch (Exception e) {
            throw new RPCException(e);
        }
    }

    public Class<?> getReqestBodyType() {
        return this.requestBodyType;
    }

    public Class<?> getResponseBodyType() {
        return this.responseBodyType;
    }

    private void parseHandler(Handler handler) {
        TypeResolver typeResolver = new TypeResolver();
        List<ResolvedType> types = typeResolver.resolve(handler.getClass()).findSupertype(Handler.class)
                .getTypeParameters();
        this.requestBodyType = types.get(0).getErasedType();
        this.responseBodyType = types.get(1).getErasedType();
        this.handler = handler;
    }

    public Handler getHandler() {
        return this.handler;
    }

    protected void validate(Request<ByteBuffer> transportRequest) throws RPCException {
//        if (!transportRequest.getEncoding().equals(getEncoding())) {
//            throw new RPCException(String.format("Content type mismatch registered handle's type."));
//        }
    }
}
