/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.encoding.raw;

import io.yarpc.RPC;
import io.yarpc.RPCFuture;
import io.yarpc.Request;
import io.yarpc.Response;
import io.yarpc.config.Config;
import io.yarpc.handler.Handler;
import io.yarpc.transport.Inbound;
import io.yarpc.transport.Outbound;
import io.yarpc.transport.http.HTTPInbound;
import io.yarpc.transport.http.HTTPOutbound;
import io.yarpc.util.BufferUtil;
import org.joda.time.Duration;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;


public class RawIntegrationTest {
    private static final String respString = "apple";
    private static final int httpPort = 9090;
    private static final String ip = "127.0.0.1";
    private static RPC rpc;

    @BeforeClass
    public static void setupServer() throws IOException {
        final Inbound httpInbound = setupHttpServer();
        final Outbound httpOutbound = setupHttpClient();
        rpc = new RPC(new Config(
                "testServer",
                new ArrayList<Inbound>() {{
                    this.add(httpInbound);
                }},
                new HashMap<String, Outbound>() {{
                    this.put("testHttpServer", httpOutbound);
                }}));
        registerEndpoint();
        rpc.start();
    }

    private static Outbound setupHttpClient() throws IOException {
        return new HTTPOutbound(ip, httpPort);
    }

    private static void registerEndpoint() {
        registerHttpEndpoint();
    }

    private static void registerHttpEndpoint() {
        RawInboundHandler rawInboundHandler = new RawInboundHandler(new Handler<ByteBuffer, ByteBuffer>() {

            @Override
            public Response<ByteBuffer> handle(Request<ByteBuffer> request) {
                return Response.Builder.forBody(request.getBody()).build();
            }
        });

        rpc.register("testHttpEndpoint", rawInboundHandler);
    }

    public static Inbound setupHttpServer() {
        return new HTTPInbound(httpPort);
    }

    @AfterClass
    public static void shutdownHttpServer() {
        rpc.stop();
    }

    @Test
    public void httpTest() throws Exception {
        RawClient client = new RawClient(rpc);
        Request request = new Request.Builder<ByteBuffer>()
                .body(ByteBuffer.wrap(respString.getBytes()))
                .service("testHttpServer")
                .procedure("testHttpEndpoint")
                .caller("testClient")
                .timeout(Duration.millis(1000))
                .build();
        RPCFuture<Response<ByteBuffer>> future = client.call(request);
        Response<ByteBuffer> response = future.get();
        Assert.assertEquals(new String(BufferUtil.toBytes(response.getBody())), respString);
    }
}
