/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.encoding.raw;

import io.yarpc.RPC;
import io.yarpc.RPCClient;
import io.yarpc.RPCFuture;
import io.yarpc.Request;
import io.yarpc.Response;
import io.yarpc.encoding.Encoding;
import io.yarpc.encoding.Serializer;
import io.yarpc.transport.Outbound;
import io.yarpc.transport.ResponseListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;

public class RawClient extends RPCClient<ByteBuffer> {
    private static final Logger logger = LoggerFactory.getLogger(RawClient.class);
    private static final Serializer<ByteBuffer> serializer = new RawSerializer();
    private final RPC rpc;

    public RawClient(RPC rpc) {
        this.rpc = rpc;
    }

    @Override
    public <I extends ByteBuffer, O extends ByteBuffer> RPCFuture<Response<O>> call(Request<I> request, Class<O>
            responseBodyType) throws Exception {
        Outbound outbound = this.rpc.lookUpOutbound(request.getService());
        request.setEncoding(Encoding.RAW);
        logger.debug(String.format("sending request %s", request.toString()));

        Request<ByteBuffer> transportRequest = Request.toByteBufferRequest(request, RawClient.serializer);
        RPCFuture<Response<O>> rpcFuture = new RPCFuture<>();
        outbound.send(transportRequest, new RawResponseListener<O>(responseBodyType, rpcFuture));
        return rpcFuture;
    }

    public <I extends ByteBuffer> RPCFuture<Response<I>> call(Request<I> request) throws Exception {
        return this.call(request, request.getGenericType());
    }

    private static class RawResponseListener<O extends ByteBuffer> implements ResponseListener {
        private final Class<O> responseBodyType;
        private final RPCFuture<Response<O>> rpcFuture;

        private RawResponseListener(Class<O> responseBodyType, RPCFuture<Response<O>> rpcFuture) {
            this.responseBodyType = responseBodyType;
            this.rpcFuture = rpcFuture;
        }

        @Override
        public void onResponseReceived(Response<ByteBuffer> transportResponse) {

            try {
                Response<O> response = Response.fromByteBufferResponse(transportResponse, RawClient.serializer,
                        this.responseBodyType);
                this.rpcFuture.set(response);
            } catch (Exception e) {
                this.rpcFuture.setException(e);
            }
        }

        @Override
        public void onErrorReceived(Response<ByteBuffer> transportResponse) {
            this.rpcFuture.setException(transportResponse.getException());
        }
    }
}
