/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc.crossdock;

import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.FullHttpResponse;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.codec.http.QueryStringDecoder;
import io.yarpc.RPCFuture;
import io.yarpc.Response;
import io.yarpc.crossdock.echo.Echo;
import io.yarpc.crossdock.echo.EchoClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.UUID;

public class HarnessServerInboundHandler extends SimpleChannelInboundHandler<FullHttpRequest> {
    private static final Logger logger = LoggerFactory.getLogger(HarnessServerInboundHandler.class);
    private static final int remotePort = 8081;

    @Override
    protected void messageReceived(ChannelHandlerContext channelHandlerContext, FullHttpRequest fullHttpRequest)
            throws Exception {
        QueryStringDecoder queryStringDecoder = new QueryStringDecoder(fullHttpRequest.uri());

        List<String> behaviors = queryStringDecoder.parameters().get("behavior");
        List<String> servers = queryStringDecoder.parameters().get("server");

        FullHttpResponse httpResponse;
        if (behaviors == null || servers == null || behaviors.size() != 1 || servers.size() != 1) {
            httpResponse = new DefaultFullHttpResponse(HttpVersion.HTTP_1_1, HttpResponseStatus.OK);
            channelHandlerContext.writeAndFlush(httpResponse);
            return;
        }

        String behavior = behaviors.get(0);
        String server = servers.get(0);

        logger.debug("behavior " + behavior + " " + server);

        String respBody = "not ok";
        if (behavior.equals("echo")) {
            EchoClient echoClient = new EchoClient(server, remotePort);
            String hash = UUID.randomUUID().toString().substring(0, 7);
            RPCFuture<Response<Echo>> future = echoClient.call(server, new Echo(hash));
            Response<Echo> response = future.get();
            if (response.getBody().getToken().equals(hash)) {
                respBody = "ok - Server said: " + hash;
            }
        }

        httpResponse = new DefaultFullHttpResponse(
                HttpVersion.HTTP_1_1,
                HttpResponseStatus.OK,
                Unpooled.wrappedBuffer(respBody.getBytes()));
        channelHandlerContext.writeAndFlush(httpResponse);
        channelHandlerContext.pipeline().channel().close();
        channelHandlerContext.close();
    }
}
