/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc;

import io.yarpc.context.Context;
import io.yarpc.encoding.Serializer;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;

public class Response<T> {
    private Map<String, String> headers;
    private T body;
    private Context context;
    private Exception exception;

    private Response(Builder<T> builder) {
        this.headers = builder.nestedHeaders;
        this.context = builder.nestedContext;
        this.body = builder.nestedRespBody;
        this.exception = builder.nestedException;
    }

    public static Response<ByteBuffer> toByteBufferResponse(Response response, Serializer serializer) throws Exception {
        return Response.Builder
                .forBody(serializer.marshal(response.getBody()))
                .headers(response.getHeaders())
                .context(response.getContext())
                .exception(response.getException())
                .build();
    }

    public static Response fromByteBufferResponse(Response<ByteBuffer> response, Serializer serializer, Class<?>
            respBodyType) throws Exception {
        return Response.Builder
                .forBody(serializer.unmarshal(response.getBody(), respBodyType))
                .headers(response.getHeaders())
                .context(response.getContext())
                .exception(response.getException())
                .build();
    }

    public Map<String, String> getHeaders() {
        if (this.headers == null) {
            this.headers = new HashMap<>();
        }
        return this.headers;
    }

    public void setHeaders(Map<String, String> headers) {
        this.headers = headers;
    }

    public Class<T> getGenericType() {
        return (Class<T>) this.body.getClass();
    }

    public T getBody() {
        return this.body;
    }

    public void setBody(T body) {
        this.body = body;
    }

    public Context getContext() {
        return this.context;
    }

    public void setContext(Context context) {
        this.context = context;
    }

    public Exception getException() {
        return this.exception;
    }

    public void setException(Exception exception) {
        this.exception = exception;
    }

    public static class Builder<T> {
        public Exception nestedException;
        private T nestedRespBody;
        private Map<String, String> nestedHeaders;
        private Context nestedContext;

        public Builder() {
        }

        public static <T> Builder<T> forBody(T body) {
            Builder<T> builder = new Builder<T>();
            return builder.body(body);
        }

        public Builder<T> body(T body) {
            this.nestedRespBody = body;
            return this;
        }

        public Builder<T> exception(Exception e) {
            this.nestedException = e;
            return this;
        }

        public Builder<T> headers(Map<String, String> headers) {
            this.nestedHeaders = headers;
            return this;
        }

        public Builder<T> context(Context context) {
            this.nestedContext = context;
            return this;
        }

        public Response<T> build() {
            return new Response<T>(this);
        }
    }
}
