/**
 * Copyright (c) 2016, Uber Technologies, Inc
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package io.yarpc;

import io.yarpc.context.Context;
import io.yarpc.encoding.Serializer;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.joda.time.Duration;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;

public class Request<T> {
    private Context context;
    private Map<String, String> headers;
    private String encoding;
    private String service;
    private String procedure;
    private Duration timeout;
    private String caller;
    private T body;

    private Request(Builder<T> builder) {
        this.body = builder.nestedRespBody;
        this.headers = builder.nestedHeaders;
        this.procedure = builder.nestedProcedure;
        this.service = builder.nestedService;
        this.context = builder.nestedContext;
        this.timeout = builder.nestedTimeout;
        this.encoding = builder.nestedEncoding;
        if (this.timeout == null) {
            this.timeout = Duration.millis(1000);
        }
        this.caller = builder.nestedCaller;
    }

    public static Request<ByteBuffer> toByteBufferRequest(Request request, Serializer serializer) throws Exception {
        return Request.Builder
                .forBody(serializer.marshal(request.getBody()))
                .caller(request.getCaller())
                .context(request.getContext())
                .encoding(request.getEncoding())
                .timeout(request.getTimeout())
                .procedure(request.getProcedure())
                .headers(request.getHeaders())
                .service(request.getService())
                .build();
    }

    public static Request fromByteBufferRequest(Request<ByteBuffer> request, Serializer serializer, Class<?> bodyType)
            throws Exception {
        return Request.Builder
                .forBody(serializer.unmarshal(request.getBody(), bodyType))
                .caller(request.getCaller())
                .context(request.getContext())
                .encoding(request.getEncoding())
                .timeout(request.getTimeout())
                .procedure(request.getProcedure())
                .headers(request.getHeaders())
                .service(request.getService())
                .build();
    }

    public Map<String, String> getHeaders() {
        if (this.headers == null) {
            this.headers = new HashMap<>();
        }
        return this.headers;
    }

    public void setHeaders(Map<String, String> headers) {
        this.headers = headers;
    }

    public String getEncoding() {
        return this.encoding;
    }

    public void setEncoding(String encoding) {
        this.encoding = encoding;
    }

    public String getService() {
        return this.service;
    }

    public void setService(String service) {
        this.service = service;
    }

    public String getProcedure() {
        return this.procedure;
    }

    public void setProcedure(String procedure) {
        this.procedure = procedure;
    }

    public Context getContext() {
        return this.context;
    }

    public void setContext(Context context) {
        this.context = context;
    }

    public String getCaller() {
        return this.caller;
    }

    public void setCaller(String caller) {
        this.caller = caller;
    }

    public Duration getTimeout() {
        return this.timeout;
    }

    public void setTimeout(Duration timeout) {
        this.timeout = timeout;
    }

    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    public Class<T> getGenericType() {
        return (Class<T>) this.body.getClass();
    }

    public T getBody() {
        return this.body;
    }

    public void setBody(T body) {
        this.body = body;
    }

    public static class Builder<T> {
        private T nestedRespBody;
        private Map<String, String> nestedHeaders;
        private String nestedProcedure;
        private String nestedService;
        private Context nestedContext;
        private Duration nestedTimeout;
        private String nestedCaller;
        private String nestedEncoding;

        public Builder() {
        }

        public static <T> Builder<T> forBody(T body) {
            Builder<T> builder = new Builder<T>();
            return builder.body(body);
        }

        public Builder<T> body(T body) {
            this.nestedRespBody = body;
            return this;
        }

        public Request<T> build() {
            return new Request<T>(this);
        }

        public Builder<T> timeout(Duration timeout) {
            this.nestedTimeout = timeout;
            return this;
        }

        public Builder<T> encoding(String encoding) {
            this.nestedEncoding = encoding;
            return this;
        }

        public Builder<T> caller(String caller) {
            this.nestedCaller = caller;
            return this;
        }

        public Builder<T> headers(Map<String, String> headers) {
            this.nestedHeaders = headers;
            return this;
        }

        public Builder<T> service(String service) {
            this.nestedService = service;
            return this;
        }

        public Builder<T> procedure(String procedure) {
            this.nestedProcedure = procedure;
            return this;
        }

        public Builder<T> context(Context context) {
            this.nestedContext = context;
            return this;
        }
    }
}
